<?php
require_once 'BaseModel.php';

class GuruModel extends BaseModel {
    
    public function __construct() {
        parent::__construct();
    }
    
    // ==================== DASHBOARD GURU ====================
    public function getDashboardStats($guruId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        $stats = [];
        
        // Total kelas yang diampu
        $sql = "SELECT COUNT(DISTINCT kelas_id) as total 
                FROM guru_mapel 
                WHERE guru_id = ? AND tahun_ajaran_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_kelas'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total mapel yang diampu
        $sql = "SELECT COUNT(DISTINCT mapel_id) as total 
                FROM guru_mapel 
                WHERE guru_id = ? AND tahun_ajaran_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_mapel'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total siswa yang diajar
        $sql = "SELECT COUNT(DISTINCT s.id) as total
                FROM siswa s
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN guru_mapel gm ON rk.kelas_id = gm.kelas_id
                WHERE gm.guru_id = ? AND gm.tahun_ajaran_id = ? AND s.status = 'aktif'";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_siswa'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total TP yang dibuat oleh guru ini
        $sql = "SELECT COUNT(DISTINCT tp.id) as total 
                FROM tujuan_pembelajaran tp
                JOIN guru_mapel gm ON (tp.mapel_id = gm.mapel_id AND tp.kelas_id = gm.kelas_id)
                WHERE gm.guru_id = ? AND tp.tahun_ajaran_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_tp'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total kegiatan formatif
        $sql = "SELECT COUNT(*) as total 
                FROM kegiatan_formatif kf
                WHERE kf.guru_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId]);
        $stats['total_kegiatan'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        return $stats;
    }
    
    public function getActiveTahunAjaran() {
        $sql = "SELECT * FROM tahun_ajaran WHERE status = 'aktif' LIMIT 1";
        $stmt = $this->executeQuery($sql);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }
    
    // ==================== GURU MAPEL MANAGEMENT ====================
    public function getGuruMapel($guruId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT gm.*, m.nama_mapel, m.kode_mapel, k.nama_kelas, k.tingkat, k.jurusan,
                       ta.tahun_ajaran, ta.semester,
                       (SELECT COUNT(*) FROM riwayat_kelas rk WHERE rk.kelas_id = k.id AND rk.tahun_ajaran_id = ?) as total_siswa,
                       (SELECT COUNT(*) FROM tujuan_pembelajaran tp WHERE tp.mapel_id = gm.mapel_id AND tp.kelas_id = gm.kelas_id AND tp.tahun_ajaran_id = ?) as total_tp
                FROM guru_mapel gm
                JOIN mata_pelajaran m ON gm.mapel_id = m.id
                JOIN kelas k ON gm.kelas_id = k.id
                JOIN tahun_ajaran ta ON gm.tahun_ajaran_id = ta.id
                WHERE gm.guru_id = ? AND ta.status = 'aktif'
                ORDER BY k.tingkat, k.nama_kelas, m.nama_mapel";
        
        $stmt = $this->executeQuery($sql, [$tahunAktif['id'], $tahunAktif['id'], $guruId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    // ==================== TUJUAN PEMBELAJARAN MANAGEMENT ====================
    public function getTPByMapel($mapelId, $kelasId = null) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        if ($kelasId) {
            // Filter berdasarkan mapel_id DAN kelas_id
            $sql = "SELECT tp.*, k.nama_kelas, k.tingkat, m.nama_mapel, m.kode_mapel,
                           (SELECT COUNT(*) FROM kegiatan_formatif kf WHERE kf.tp_id = tp.id) as total_kegiatan,
                           (SELECT COUNT(*) FROM nilai_formatif nf WHERE nf.tp_id = tp.id) as total_nilai_formatif,
                           (SELECT COUNT(*) FROM nilai_sumatif ns WHERE ns.tp_id = tp.id) as total_nilai_sumatif
                    FROM tujuan_pembelajaran tp
                    JOIN kelas k ON tp.kelas_id = k.id
                    JOIN mata_pelajaran m ON tp.mapel_id = m.id
                    WHERE tp.mapel_id = ? AND tp.kelas_id = ? AND tp.tahun_ajaran_id = ?
                    ORDER BY tp.created_at DESC";
            $stmt = $this->executeQuery($sql, [$mapelId, $kelasId, $tahunAktif['id']]);
        } else {
            // Fallback: ambil semua TP untuk mapel tersebut
            $sql = "SELECT tp.*, k.nama_kelas, k.tingkat, m.nama_mapel, m.kode_mapel,
                           (SELECT COUNT(*) FROM kegiatan_formatif kf WHERE kf.tp_id = tp.id) as total_kegiatan,
                           (SELECT COUNT(*) FROM nilai_formatif nf WHERE nf.tp_id = tp.id) as total_nilai_formatif,
                           (SELECT COUNT(*) FROM nilai_sumatif ns WHERE ns.tp_id = tp.id) as total_nilai_sumatif
                    FROM tujuan_pembelajaran tp
                    JOIN kelas k ON tp.kelas_id = k.id
                    JOIN mata_pelajaran m ON tp.mapel_id = m.id
                    WHERE tp.mapel_id = ? AND tp.tahun_ajaran_id = ?
                    ORDER BY tp.created_at DESC";
            $stmt = $this->executeQuery($sql, [$mapelId, $tahunAktif['id']]);
        }
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function getTPById($tpId) {
        $sql = "SELECT tp.*, m.nama_mapel, m.kode_mapel, k.nama_kelas, k.tingkat, k.jurusan,
                       ta.tahun_ajaran, ta.semester,
                       (SELECT COUNT(*) FROM kegiatan_formatif kf WHERE kf.tp_id = tp.id) as total_kegiatan
                FROM tujuan_pembelajaran tp
                JOIN mata_pelajaran m ON tp.mapel_id = m.id
                JOIN kelas k ON tp.kelas_id = k.id
                JOIN tahun_ajaran ta ON tp.tahun_ajaran_id = ta.id
                WHERE tp.id = ?";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }
    
    public function createTP($data) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "INSERT INTO tujuan_pembelajaran (mapel_id, kelas_id, tahun_ajaran_id, kode_tp, deskripsi, status) 
                VALUES (?, ?, ?, ?, ?, 'aktif')";
        
        $result = $this->executeQuery($sql, [
            $data['mapel_id'], 
            $data['kelas_id'],
            $tahunAktif['id'], 
            $data['kode_tp'], 
            $data['deskripsi']
        ]);
        
        return $result ? $this->db->lastInsertId() : false;
    }
    
    public function updateTP($tpId, $data) {
        $sql = "UPDATE tujuan_pembelajaran 
                SET kode_tp = ?, deskripsi = ?, status = ?
                WHERE id = ?";
        
        return $this->executeQuery($sql, [
            $data['kode_tp'],
            $data['deskripsi'],
            $data['status'],
            $tpId
        ]);
    }

    public function deleteTP($tpId) {
        // Cek apakah TP memiliki data terkait
        $checkSql = "SELECT 
                    (SELECT COUNT(*) FROM kegiatan_formatif WHERE tp_id = ?) as total_kegiatan,
                    (SELECT COUNT(*) FROM nilai_formatif WHERE tp_id = ?) as total_nilai_formatif,
                    (SELECT COUNT(*) FROM nilai_sumatif WHERE tp_id = ?) as total_nilai_sumatif";
        
        $checkStmt = $this->executeQuery($checkSql, [$tpId, $tpId, $tpId]);
        $checkData = $checkStmt ? $checkStmt->fetch(PDO::FETCH_ASSOC) : [];
        
        $totalRelated = array_sum($checkData);
        
        if ($totalRelated > 0) {
            // Jika ada data terkait, kembalikan false
            return false;
        }
        
        // Hapus TP jika tidak ada data terkait
        $sql = "DELETE FROM tujuan_pembelajaran WHERE id = ?";
        return $this->executeQuery($sql, [$tpId]);
    }

    // ==================== KEGIATAN FORMATIF MANAGEMENT ====================
    public function getKegiatanFormatifByTP($tpId) {
        $sql = "SELECT kf.*, u.nama as nama_guru,
                       (SELECT COUNT(*) FROM nilai_formatif nf WHERE nf.kegiatan_formatif_id = kf.id) as total_nilai
                FROM kegiatan_formatif kf
                JOIN guru g ON kf.guru_id = g.id
                JOIN users u ON g.user_id = u.id
                WHERE kf.tp_id = ?
                ORDER BY kf.tanggal DESC, kf.created_at DESC";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function createKegiatanFormatif($data) {
        $sql = "INSERT INTO kegiatan_formatif (tp_id, guru_id, nama_kegiatan, jenis_kegiatan, deskripsi, bobot, tanggal, status) 
                VALUES (?, ?, ?, ?, ?, ?, ?, 'aktif')";
        
        return $this->executeQuery($sql, [
            $data['tp_id'],
            $data['guru_id'],
            $data['nama_kegiatan'],
            $data['jenis_kegiatan'],
            $data['deskripsi'],
            $data['bobot'],
            $data['tanggal']
        ]);
    }

    public function getKegiatanFormatifById($kegiatanId) {
        $sql = "SELECT kf.*, tp.kode_tp, tp.deskripsi as deskripsi_tp, 
                       m.nama_mapel, m.kode_mapel, u.nama as nama_guru,
                       k.nama_kelas, k.tingkat
                FROM kegiatan_formatif kf
                JOIN tujuan_pembelajaran tp ON kf.tp_id = tp.id
                JOIN mata_pelajaran m ON tp.mapel_id = m.id
                JOIN kelas k ON tp.kelas_id = k.id
                JOIN guru g ON kf.guru_id = g.id
                JOIN users u ON g.user_id = u.id
                WHERE kf.id = ?";
        
        $stmt = $this->executeQuery($sql, [$kegiatanId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }

    public function updateKegiatanFormatif($kegiatanId, $data) {
        $sql = "UPDATE kegiatan_formatif SET 
                nama_kegiatan = ?, jenis_kegiatan = ?, deskripsi = ?, 
                bobot = ?, tanggal = ?, status = ?, updated_at = NOW()
                WHERE id = ?";
        
        return $this->executeQuery($sql, [
            $data['nama_kegiatan'],
            $data['jenis_kegiatan'],
            $data['deskripsi'],
            $data['bobot'],
            $data['tanggal'],
            $data['status'],
            $kegiatanId
        ]);
    }

    public function deleteKegiatanFormatif($kegiatanId) {
        // Cek apakah kegiatan memiliki nilai
        $checkSql = "SELECT COUNT(*) as total_nilai FROM nilai_formatif WHERE kegiatan_formatif_id = ?";
        $checkStmt = $this->executeQuery($checkSql, [$kegiatanId]);
        $checkData = $checkStmt ? $checkStmt->fetch(PDO::FETCH_ASSOC) : [];
        
        if ($checkData['total_nilai'] > 0) {
            return false;
        }
        
        $sql = "DELETE FROM kegiatan_formatif WHERE id = ?";
        return $this->executeQuery($sql, [$kegiatanId]);
    }

    // ==================== NILAI FORMATIF TERHUBUNG KEGIATAN ====================
    public function inputNilaiFormatifWithKegiatan($data) {
        $checkSql = "SELECT id FROM nilai_formatif 
                     WHERE kegiatan_formatif_id = ? AND siswa_id = ?";
        $checkStmt = $this->executeQuery($checkSql, [
            $data['kegiatan_formatif_id'], $data['siswa_id']
        ]);
        
        if ($checkStmt && $checkStmt->rowCount() > 0) {
            $sql = "UPDATE nilai_formatif 
                    SET nilai = ?, deskripsi = ?, updated_by = ?, updated_at = NOW() 
                    WHERE kegiatan_formatif_id = ? AND siswa_id = ?";
            $params = [
                $data['nilai'], $data['deskripsi'], $data['updated_by'],
                $data['kegiatan_formatif_id'], $data['siswa_id']
            ];
        } else {
            $sql = "INSERT INTO nilai_formatif (kegiatan_formatif_id, tp_id, siswa_id, jenis, nilai, deskripsi, updated_by) 
                    VALUES (?, ?, ?, ?, ?, ?, ?)";
            $params = [
                $data['kegiatan_formatif_id'], $data['tp_id'], $data['siswa_id'],
                $data['jenis'], $data['nilai'], $data['deskripsi'], $data['updated_by']
            ];
        }
        
        return $this->executeQuery($sql, $params);
    }

    public function getNilaiFormatifByKegiatan($kegiatanId) {
        $sql = "SELECT nf.*, s.nis, u.nama as nama_siswa, k.nama_kelas
                FROM nilai_formatif nf
                JOIN siswa s ON nf.siswa_id = s.id
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN kelas k ON rk.kelas_id = k.id
                JOIN tahun_ajaran ta ON rk.tahun_ajaran_id = ta.id
                WHERE nf.kegiatan_formatif_id = ? AND ta.status = 'aktif'
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql, [$kegiatanId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function getNilaiFormatifBySiswaTP($siswaId, $tpId) {
        $sql = "SELECT nf.*, kf.nama_kegiatan, kf.jenis_kegiatan, kf.bobot, kf.tanggal
                FROM nilai_formatif nf
                JOIN kegiatan_formatif kf ON nf.kegiatan_formatif_id = kf.id
                WHERE nf.siswa_id = ? AND kf.tp_id = ?
                ORDER BY kf.tanggal";
        
        $stmt = $this->executeQuery($sql, [$siswaId, $tpId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    // ==================== NILAI MANAGEMENT (LEGACY) ====================
    public function inputNilaiFormatif($data) {
        $checkSql = "SELECT id FROM nilai_formatif 
                     WHERE tp_id = ? AND siswa_id = ? AND jenis = ?";
        $checkStmt = $this->executeQuery($checkSql, [
            $data['tp_id'], $data['siswa_id'], $data['jenis']
        ]);
        
        if ($checkStmt && $checkStmt->rowCount() > 0) {
            $sql = "UPDATE nilai_formatif 
                    SET nilai = ?, deskripsi = ?, updated_by = ?, updated_at = NOW() 
                    WHERE tp_id = ? AND siswa_id = ? AND jenis = ?";
        } else {
            $sql = "INSERT INTO nilai_formatif (tp_id, siswa_id, jenis, nilai, deskripsi, updated_by) 
                    VALUES (?, ?, ?, ?, ?, ?)";
        }
        
        return $this->executeQuery($sql, [
            $data['nilai'], $data['deskripsi'], $data['updated_by'],
            $data['tp_id'], $data['siswa_id'], $data['jenis']
        ]);
    }
    
    public function getNilaiSumatifTP($tpId, $siswaId = null) {
        $sql = "SELECT ns.*, u.nama as nama_siswa, s.nis, k.nama_kelas
                FROM nilai_sumatif ns
                JOIN siswa s ON ns.siswa_id = s.id
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN kelas k ON rk.kelas_id = k.id
                JOIN tahun_ajaran ta ON rk.tahun_ajaran_id = ta.id
                WHERE ns.tp_id = ? AND ta.status = 'aktif'" . 
                ($siswaId ? " AND ns.siswa_id = ?" : "") . "
                ORDER BY u.nama";
        
        $params = [$tpId];
        if ($siswaId) {
            $params[] = $siswaId;
        }
        
        $stmt = $this->executeQuery($sql, $params);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    public function inputNilaiSumatifTP($data) {
        $checkSql = "SELECT id FROM nilai_sumatif 
                     WHERE tp_id = ? AND siswa_id = ?";
        $checkStmt = $this->executeQuery($checkSql, [
            $data['tp_id'], $data['siswa_id']
        ]);
        
        if ($checkStmt && $checkStmt->rowCount() > 0) {
            $sql = "UPDATE nilai_sumatif 
                    SET nilai = ?, jenis_penilaian = ?, deskripsi = ?, updated_by = ?, updated_at = NOW() 
                    WHERE tp_id = ? AND siswa_id = ?";
        } else {
            $sql = "INSERT INTO nilai_sumatif (tp_id, siswa_id, nilai, jenis_penilaian, deskripsi, updated_by) 
                    VALUES (?, ?, ?, ?, ?, ?)";
        }
        
        return $this->executeQuery($sql, [
            $data['nilai'], $data['jenis_penilaian'], $data['deskripsi'], $data['updated_by'],
            $data['tp_id'], $data['siswa_id']
        ]);
    }
    
    public function getNilaiSumatifAkhir($mapelId, $kelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT nsa.*, u.nama as nama_siswa, s.nis
                FROM nilai_sumatif_akhir nsa
                JOIN siswa s ON nsa.siswa_id = s.id
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                WHERE nsa.mapel_id = ? AND nsa.tahun_ajaran_id = ? 
                AND rk.kelas_id = ? AND rk.tahun_ajaran_id = ?
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql, [
            $mapelId, $tahunAktif['id'], $kelasId, $tahunAktif['id']
        ]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    public function inputNilaiSumatifAkhir($data) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $checkSql = "SELECT id FROM nilai_sumatif_akhir 
                     WHERE siswa_id = ? AND mapel_id = ? AND tahun_ajaran_id = ?";
        $checkStmt = $this->executeQuery($checkSql, [
            $data['siswa_id'], $data['mapel_id'], $tahunAktif['id']
        ]);
        
        if ($checkStmt && $checkStmt->rowCount() > 0) {
            $sql = "UPDATE nilai_sumatif_akhir 
                    SET nilai = ?, updated_by = ?, updated_at = NOW() 
                    WHERE siswa_id = ? AND mapel_id = ? AND tahun_ajaran_id = ?";
        } else {
            $sql = "INSERT INTO nilai_sumatif_akhir (siswa_id, mapel_id, tahun_ajaran_id, semester, nilai, updated_by) 
                    VALUES (?, ?, ?, ?, ?, ?)";
        }
        
        return $this->executeQuery($sql, [
            $data['nilai'], $data['updated_by'],
            $data['siswa_id'], $data['mapel_id'], $tahunAktif['id'], 
            $tahunAktif['semester']
        ]);
    }

    // ==================== REKAP NILAI TERPADU ====================
    public function getRekapNilaiTerpadu($kelasId, $mapelId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT 
                    s.id as siswa_id, u.nama as nama_siswa, s.nis,
                    tp.id as tp_id, tp.kode_tp, tp.deskripsi as deskripsi_tp,
                    kf.nama_kegiatan, kf.jenis_kegiatan, kf.bobot as bobot_kegiatan,
                    nf.nilai as nilai_formatif, nf.deskripsi as deskripsi_formatif,
                    ns.nilai as nilai_sumatif_tp, ns.deskripsi as deskripsi_sumatif,
                    nsa.nilai as nilai_sumatif_akhir
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN tujuan_pembelajaran tp ON tp.mapel_id = ? AND tp.tahun_ajaran_id = ? AND tp.kelas_id = ?
                LEFT JOIN kegiatan_formatif kf ON kf.tp_id = tp.id
                LEFT JOIN nilai_formatif nf ON (nf.kegiatan_formatif_id = kf.id AND nf.siswa_id = s.id)
                LEFT JOIN nilai_sumatif ns ON (tp.id = ns.tp_id AND s.id = ns.siswa_id)
                LEFT JOIN nilai_sumatif_akhir nsa ON (s.id = nsa.siswa_id AND nsa.mapel_id = ? AND nsa.tahun_ajaran_id = ?)
                WHERE rk.kelas_id = ? AND rk.tahun_ajaran_id = ? AND s.status = 'aktif'
                ORDER BY u.nama, tp.kode_tp, kf.nama_kegiatan";
        
        $stmt = $this->executeQuery($sql, [
            $mapelId, $tahunAktif['id'], $kelasId,
            $mapelId, $tahunAktif['id'],
            $kelasId, $tahunAktif['id']
        ]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function getRekapNilaiKelas($kelasId, $mapelId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT s.id as siswa_id, u.nama as nama_siswa, s.nis,
                       tp.kode_tp, tp.deskripsi as deskripsi_tp,
                       nf.jenis, nf.nilai as nilai_formatif, nf.deskripsi as deskripsi_formatif,
                       ns.nilai as nilai_sumatif_tp, ns.deskripsi as deskripsi_sumatif,
                       nsa.nilai as nilai_sumatif_akhir
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN tujuan_pembelajaran tp ON tp.mapel_id = ? AND tp.tahun_ajaran_id = ? AND tp.kelas_id = ?
                LEFT JOIN nilai_formatif nf ON (tp.id = nf.tp_id AND s.id = nf.siswa_id)
                LEFT JOIN nilai_sumatif ns ON (tp.id = ns.tp_id AND s.id = ns.siswa_id)
                LEFT JOIN nilai_sumatif_akhir nsa ON (s.id = nsa.siswa_id AND nsa.mapel_id = ? AND nsa.tahun_ajaran_id = ?)
                WHERE rk.kelas_id = ? AND rk.tahun_ajaran_id = ? AND s.status = 'aktif'
                ORDER BY u.nama, tp.kode_tp, nf.jenis";
        
        $stmt = $this->executeQuery($sql, [
            $mapelId, $tahunAktif['id'], $kelasId,
            $mapelId, $tahunAktif['id'],
            $kelasId, $tahunAktif['id']
        ]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    // ==================== SISWA MANAGEMENT ====================
    public function getSiswaByKelas($kelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT s.*, u.nama, u.username, k.nama_kelas, k.tingkat
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN kelas k ON rk.kelas_id = k.id
                WHERE rk.kelas_id = ? AND rk.tahun_ajaran_id = ? AND s.status = 'aktif'
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql, [$kelasId, $tahunAktif['id']]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    // ==================== WALI KELAS MANAGEMENT ====================
    public function getWaliKelasByGuru($guruId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT wk.*, k.nama_kelas, k.tingkat, k.jurusan, ta.tahun_ajaran, ta.semester,
                       (SELECT COUNT(*) FROM riwayat_kelas rk WHERE rk.kelas_id = k.id AND rk.tahun_ajaran_id = ?) as total_siswa
                FROM wali_kelas wk
                JOIN kelas k ON wk.kelas_id = k.id
                JOIN tahun_ajaran ta ON wk.tahun_ajaran_id = ta.id
                WHERE wk.guru_id = ? AND ta.status = 'aktif'";
        
        $stmt = $this->executeQuery($sql, [$tahunAktif['id'], $guruId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }

    // ==================== VALIDASI AKSES GURU ====================
    public function validateGuruMapelAccess($guruId, $mapelId, $kelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT id FROM guru_mapel 
                WHERE guru_id = ? AND mapel_id = ? AND kelas_id = ? AND tahun_ajaran_id = ?";
        
        $stmt = $this->executeQuery($sql, [
            $guruId, $mapelId, $kelasId, $tahunAktif['id']
        ]);
        
        return $stmt && $stmt->rowCount() > 0;
    }

    public function validateGuruTPAccess($guruId, $tpId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT gm.id 
                FROM guru_mapel gm
                JOIN tujuan_pembelajaran tp ON (gm.mapel_id = tp.mapel_id AND gm.kelas_id = tp.kelas_id)
                WHERE gm.guru_id = ? AND tp.id = ? AND gm.tahun_ajaran_id = ?";
        
        $stmt = $this->executeQuery($sql, [$guruId, $tpId, $tahunAktif['id']]);
        return $stmt && $stmt->rowCount() > 0;
    }

    public function validateGuruKegiatanAccess($guruId, $kegiatanId) {
        $sql = "SELECT kf.id 
                FROM kegiatan_formatif kf
                WHERE kf.id = ? AND kf.guru_id = ?";
        
        $stmt = $this->executeQuery($sql, [$kegiatanId, $guruId]);
        return $stmt && $stmt->rowCount() > 0;
    }

    // ==================== UTILITY METHODS ====================
    public function getRataRataNilaiFormatifByTP($tpId) {
        $sql = "SELECT AVG(nf.nilai) as rata_rata, COUNT(nf.id) as total_nilai
                FROM nilai_formatif nf
                WHERE nf.tp_id = ?";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : ['rata_rata' => 0, 'total_nilai' => 0];
    }

    public function getRataRataNilaiSumatifByTP($tpId) {
        $sql = "SELECT AVG(ns.nilai) as rata_rata, COUNT(ns.id) as total_nilai
                FROM nilai_sumatif ns
                WHERE ns.tp_id = ?";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : ['rata_rata' => 0, 'total_nilai' => 0];
    }

    public function getProgressTP($tpId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT 
                    (SELECT COUNT(DISTINCT s.id) 
                     FROM siswa s 
                     JOIN riwayat_kelas rk ON s.id = rk.siswa_id 
                     JOIN tujuan_pembelajaran tp ON rk.kelas_id = tp.kelas_id 
                     WHERE tp.id = ? AND rk.tahun_ajaran_id = ?) as total_siswa,
                    (SELECT COUNT(DISTINCT nf.siswa_id) 
                     FROM nilai_formatif nf 
                     WHERE nf.tp_id = ?) as siswa_terinput_formatif,
                    (SELECT COUNT(DISTINCT ns.siswa_id) 
                     FROM nilai_sumatif ns 
                     WHERE ns.tp_id = ?) as siswa_terinput_sumatif";
        
        $stmt = $this->executeQuery($sql, [$tpId, $tahunAktif['id'], $tpId, $tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : [
            'total_siswa' => 0,
            'siswa_terinput_formatif' => 0,
            'siswa_terinput_sumatif' => 0
        ];
    }
}
?>