<?php
require_once __DIR__ . '/../../app/config/config.php';
Session::checkAuth();
Session::checkRole(['guru']);

// Error reporting untuk debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

try {
    $guruModel = new GuruModel();
    $userData = Session::getUserData();
    $guruId = $userData['guru_id'];

    // Get tahun ajaran aktif
    $tahunAktif = $guruModel->getActiveTahunAjaran();
    if (!$tahunAktif) {
        throw new Exception("Tahun ajaran aktif tidak ditemukan");
    }
    $tahun_ajaran_id = $tahunAktif['id'];

    // Get mapel yang diampu guru
    $mapel_list = $guruModel->getMapelByGuru($guruId);
    if ($mapel_list === false) {
        throw new Exception("Gagal mengambil data mata pelajaran");
    }

    // Get kelas berdasarkan mapel yang dipilih
    $selected_mapel_id = $_GET['mapel_id'] ?? null;
    $kelas_list = [];
    if ($selected_mapel_id) {
        $kelas_list = $guruModel->getKelasByMapelGuru($guruId, $selected_mapel_id);
        if ($kelas_list === false) {
            throw new Exception("Gagal mengambil data kelas");
        }
    }

    // Redirect ke halaman detail jika mapel dan kelas sudah dipilih
    if ($selected_mapel_id && isset($_GET['kelas_id'])) {
        $kelas_id = $_GET['kelas_id'];
        header("Location: nilai_sumatif_detail.php?mapel_id=$selected_mapel_id&kelas_id=$kelas_id&tahun_ajaran_id=$tahun_ajaran_id");
        exit();
    }

} catch (Exception $e) {
    // Log error dan tampilkan pesan user-friendly
    error_log("Error in rekap_nilai.php: " . $e->getMessage());
    die("Terjadi kesalahan sistem. Silakan hubungi administrator.");
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rekap Nilai Sumatif - Sistem Penilaian</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .card {
            border-radius: 12px;
            border: none;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .btn {
            border-radius: 8px;
            font-weight: 500;
        }
        
        .selection-card {
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        .selection-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 15px rgba(0, 0, 0, 0.15);
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem 1rem;
            color: #6c757d;
        }
        
        .empty-state i {
            font-size: 4rem;
            margin-bottom: 1rem;
        }
        
        .config-panel {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
        }
        
        .percentage-input {
            max-width: 120px;
        }
        
        .total-percentage {
            font-weight: bold;
            padding: 0.5rem;
            border-radius: 4px;
        }
        
        .total-percentage.valid {
            background-color: #d1edff;
            color: #0d6efd;
        }
        
        .total-percentage.invalid {
            background-color: #f8d7da;
            color: #dc3545;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <!-- Header -->
        <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
            <div class="container">
                <a class="navbar-brand" href="#">
                    <i class="bi bi-table"></i> Rekap Nilai Sumatif
                </a>
                <div class="navbar-nav ms-auto">
                    <span class="navbar-text me-3">
                        <i class="bi bi-person-circle"></i> <?= htmlspecialchars($userData['nama'] ?? 'User') ?>
                    </span>
                    <a href="dashboard.php" class="btn btn-outline-light btn-sm">
                        <i class="bi bi-house"></i> Dashboard
                    </a>
                </div>
            </div>
        </nav>

        <div class="container mt-4">
            <!-- Breadcrumb -->
            <nav aria-label="breadcrumb" class="mb-4">
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                    <li class="breadcrumb-item active">Rekap Nilai Sumatif</li>
                </ol>
            </nav>

            <!-- Error Alert (jika ada) -->
            <?php if (isset($error_message)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bi bi-exclamation-triangle-fill"></i> <?= $error_message ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <div class="row justify-content-center">
                <div class="col-md-10">
                    <!-- Konfigurasi Presentase -->
                    <div class="card mb-4">
                        <div class="card-header bg-info text-white">
                            <h5 class="mb-0">
                                <i class="bi bi-sliders"></i> Konfigurasi Presentase Nilai
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="config-panel">
                                <div class="row align-items-center">
                                    <div class="col-md-4">
                                        <label class="form-label">
                                            <i class="bi bi-percent"></i> Presentase Nilai SAS
                                        </label>
                                        <div class="input-group">
                                            <input type="number" 
                                                   class="form-control percentage-input" 
                                                   id="presentase_sas" 
                                                   min="0" 
                                                   max="100" 
                                                   value="60">
                                            <span class="input-group-text">%</span>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">
                                            <i class="bi bi-graph-up"></i> Presentase Rata-rata TP
                                        </label>
                                        <div class="input-group">
                                            <input type="number" 
                                                   class="form-control percentage-input" 
                                                   id="presentase_tp" 
                                                   min="0" 
                                                   max="100" 
                                                   value="40">
                                            <span class="input-group-text">%</span>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Total Presentase</label>
                                        <div id="total-presentase" class="total-percentage valid">
                                            100%
                                        </div>
                                        <small class="form-text text-muted">
                                            Total harus tepat 100%
                                        </small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="alert alert-info">
                                <small>
                                    <i class="bi bi-info-circle"></i> 
                                    <strong>Rumus Nilai Akhir:</strong> (SAS × Presentase SAS) + (Rata-rata TP × Presentase TP)
                                </small>
                            </div>
                        </div>
                    </div>

                    <!-- Pilihan Mapel dan Kelas -->
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">
                                <i class="bi bi-table"></i> Pilih Mata Pelajaran dan Kelas
                            </h5>
                        </div>
                        <div class="card-body">
                            <?php if (empty($mapel_list)): ?>
                                <!-- Empty State -->
                                <div class="empty-state">
                                    <i class="bi bi-book"></i>
                                    <h4>Belum Ada Mata Pelajaran</h4>
                                    <p>Anda belum diassign ke mata pelajaran apapun.</p>
                                    <a href="dashboard.php" class="btn btn-primary">
                                        <i class="bi bi-arrow-left"></i> Kembali ke Dashboard
                                    </a>
                                </div>
                            <?php else: ?>
                                <form method="GET" action="rekap_nilai.php" id="selectionForm">
                                    <input type="hidden" name="presentase_sas" id="hidden_presentase_sas" value="60">
                                    <input type="hidden" name="presentase_tp" id="hidden_presentase_tp" value="40">
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="mapel_id" class="form-label">
                                                    <i class="bi bi-book"></i> Mata Pelajaran
                                                </label>
                                                <select class="form-select" id="mapel_id" name="mapel_id" required 
                                                        onchange="this.form.submit()">
                                                    <option value="">Pilih Mata Pelajaran</option>
                                                    <?php foreach ($mapel_list as $mapel): ?>
                                                        <option value="<?= htmlspecialchars($mapel['id']) ?>" 
                                                            <?= $selected_mapel_id == $mapel['id'] ? 'selected' : '' ?>>
                                                            <?= htmlspecialchars($mapel['nama_mapel']) ?> 
                                                            (<?= htmlspecialchars($mapel['kode_mapel'] ?? '') ?>)
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="kelas_id" class="form-label">
                                                    <i class="bi bi-people"></i> Kelas
                                                </label>
                                                <select class="form-select" id="kelas_id" name="kelas_id" required
                                                        <?= empty($kelas_list) ? 'disabled' : '' ?>>
                                                    <option value="">Pilih Kelas</option>
                                                    <?php foreach ($kelas_list as $kelas): ?>
                                                        <option value="<?= htmlspecialchars($kelas['id']) ?>">
                                                            <?= htmlspecialchars($kelas['nama_kelas']) ?> 
                                                            (<?= htmlspecialchars($kelas['tingkat'] ?? '') ?>)
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                                <?php if ($selected_mapel_id && empty($kelas_list)): ?>
                                                    <div class="form-text text-warning">
                                                        <i class="bi bi-info-circle"></i> 
                                                        Tidak ada kelas yang tersedia untuk mata pelajaran ini.
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <?php if (!empty($kelas_list)): ?>
                                    <div class="text-center mt-4">
                                        <button type="submit" class="btn btn-primary btn-lg" id="submitBtn">
                                            <i class="bi bi-arrow-right-circle"></i> Lihat Rekap Nilai dengan Analisis
                                        </button>
                                    </div>
                                    <?php endif; ?>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Quick Stats -->
                    <?php if ($selected_mapel_id && !empty($kelas_list)): ?>
                    <div class="row mt-4">
                        <div class="col-md-3">
                            <div class="card selection-card text-center">
                                <div class="card-body">
                                    <i class="bi bi-book fs-1 text-primary"></i>
                                    <h5 class="mt-2"><?= count($mapel_list) ?></h5>
                                    <p class="text-muted mb-0">Mata Pelajaran</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card selection-card text-center">
                                <div class="card-body">
                                    <i class="bi bi-people fs-1 text-success"></i>
                                    <h5 class="mt-2"><?= count($kelas_list) ?></h5>
                                    <p class="text-muted mb-0">Kelas Tersedia</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card selection-card text-center">
                                <div class="card-body">
                                    <i class="bi bi-clock fs-1 text-warning"></i>
                                    <h5 class="mt-2"><?= htmlspecialchars($tahunAktif['tahun_ajaran'] ?? 'N/A') ?></h5>
                                    <p class="text-muted mb-0">Tahun Ajaran</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card selection-card text-center">
                                <div class="card-body">
                                    <i class="bi bi-percent fs-1 text-info"></i>
                                    <h5 class="mt-2" id="config-display">60:40</h5>
                                    <p class="text-muted mb-0">Konfigurasi SAS:TP</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const mapelSelect = document.getElementById('mapel_id');
            const kelasSelect = document.getElementById('kelas_id');
            const presentaseSAS = document.getElementById('presentase_sas');
            const presentaseTP = document.getElementById('presentase_tp');
            const totalPresentase = document.getElementById('total-presentase');
            const hiddenPresentaseSAS = document.getElementById('hidden_presentase_sas');
            const hiddenPresentaseTP = document.getElementById('hidden_presentase_tp');
            const configDisplay = document.getElementById('config-display');
            const submitBtn = document.getElementById('submitBtn');
            const selectionForm = document.getElementById('selectionForm');

            // Enable kelas select when mapel is selected
            if (mapelSelect && mapelSelect.value) {
                kelasSelect.disabled = false;
            }

            // Function to calculate total percentage
            function calculateTotal() {
                const sas = parseInt(presentaseSAS.value) || 0;
                const tp = parseInt(presentaseTP.value) || 0;
                const total = sas + tp;
                
                // Update total display
                totalPresentase.textContent = total + '%';
                
                // Update config display
                if (configDisplay) {
                    configDisplay.textContent = sas + ':' + tp;
                }
                
                // Validate total
                if (total === 100) {
                    totalPresentase.className = 'total-percentage valid';
                    if (submitBtn) submitBtn.disabled = false;
                    return true;
                } else {
                    totalPresentase.className = 'total-percentage invalid';
                    if (submitBtn) submitBtn.disabled = true;
                    return false;
                }
            }

            // Update hidden fields and validate on percentage change
            function updatePercentages() {
                const sas = parseInt(presentaseSAS.value) || 0;
                const tp = parseInt(presentaseTP.value) || 0;
                
                hiddenPresentaseSAS.value = sas;
                hiddenPresentaseTP.value = tp;
                
                calculateTotal();
            }

            // Event listeners for percentage inputs
            if (presentaseSAS && presentaseTP) {
                presentaseSAS.addEventListener('input', updatePercentages);
                presentaseTP.addEventListener('input', updatePercentages);
                
                // Initial calculation
                updatePercentages();
            }

            // Form submission validation
            if (selectionForm) {
                selectionForm.addEventListener('submit', function(e) {
                    if (!calculateTotal()) {
                        e.preventDefault();
                        alert('Total presentase harus tepat 100%!');
                        return false;
                    }
                    
                    // Show loading state
                    if (submitBtn) {
                        submitBtn.innerHTML = '<i class="bi bi-hourglass-split"></i> Memproses...';
                        submitBtn.disabled = true;
                    }
                });
            }

            // Auto-adjust percentages
            function autoAdjustPercentages(changedField) {
                const sas = parseInt(presentaseSAS.value) || 0;
                const tp = parseInt(presentaseTP.value) || 0;
                
                if (changedField === 'sas') {
                    presentaseTP.value = 100 - sas;
                } else {
                    presentaseSAS.value = 100 - tp;
                }
                
                updatePercentages();
            }

            // Add auto-adjust on blur
            if (presentaseSAS && presentaseTP) {
                presentaseSAS.addEventListener('blur', function() {
                    if (this.value && parseInt(this.value) >= 0 && parseInt(this.value) <= 100) {
                        autoAdjustPercentages('sas');
                    }
                });
                
                presentaseTP.addEventListener('blur', function() {
                    if (this.value && parseInt(this.value) >= 0 && parseInt(this.value) <= 100) {
                        autoAdjustPercentages('tp');
                    }
                });
            }
        });
    </script>
</body>
</html>