<?php
require_once __DIR__ . '/../../app/config/config.php';
Session::checkAuth();
Session::checkRole(['guru']);

$guruModel = new GuruModel();
$userData = Session::getUserData();
$guruId = $userData['guru_id'];

// Get parameters
$mapel_id = $_GET['mapel_id'] ?? null;
$kelas_id = $_GET['kelas_id'] ?? null;
$tahun_ajaran_id = $_GET['tahun_ajaran_id'] ?? null;

// Jika tidak ada parameter, redirect ke halaman TP
if (!$mapel_id || !$kelas_id) {
    header("Location: tp.php");
    exit();
}

// Get tahun ajaran aktif jika tidak ditentukan
if (!$tahun_ajaran_id) {
    $tahunAktif = $guruModel->getActiveTahunAjaran();
    $tahun_ajaran_id = $tahunAktif['id'];
}

// Validasi akses guru
if (!$guruModel->validateGuruMapelAccess($guruId, $mapel_id, $kelas_id)) {
    die("Akses ditolak untuk mapel dan kelas ini.");
}

// Process form submission untuk SAS
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_sas'])) {
    // Debug POST data
    echo "<!-- POST Data: " . print_r($_POST, true) . " -->\n";
    
    $success_count = 0;
    $error_count = 0;
    
    if (isset($_POST['nilai_sas']) && is_array($_POST['nilai_sas'])) {
        foreach ($_POST['nilai_sas'] as $siswa_id => $nilai_sas) {
            // Hanya proses jika nilai tidak kosong
            if ($nilai_sas !== '' && is_numeric($nilai_sas)) {
                $data = [
                    'siswa_id' => intval($siswa_id),
                    'mapel_id' => intval($mapel_id),
                    'nilai' => floatval($nilai_sas),
                    'updated_by' => $guruId
                ];
                
                echo "<!-- SAS Data for siswa $siswa_id: " . print_r($data, true) . " -->\n";
                
                if ($guruModel->inputNilaiSumatifAkhir($data)) {
                    $success_count++;
                    echo "<!-- Success for siswa $siswa_id -->\n";
                } else {
                    $error_count++;
                    echo "<!-- Failed for siswa $siswa_id -->\n";
                }
            } else {
                echo "<!-- Skipped siswa $siswa_id - nilai kosong atau invalid -->\n";
            }
        }
        
        if ($success_count > 0) {
            $_SESSION['flash_success'] = "Berhasil menyimpan $success_count nilai SAS!";
            if ($error_count > 0) {
                $_SESSION['flash_success'] .= " ($error_count gagal)";
            }
        } else {
            $_SESSION['flash_error'] = 'Tidak ada nilai SAS yang berhasil disimpan! Pastikan:';
            $_SESSION['flash_error'] .= '<br>- Nilai SAS harus diisi (tidak boleh kosong)';
            $_SESSION['flash_error'] .= '<br>- Nilai harus berupa angka antara 0-100';
        }
        
        header("Location: nilai_sumatif_detail.php?mapel_id=$mapel_id&kelas_id=$kelas_id&tahun_ajaran_id=$tahun_ajaran_id");
        exit();
    } else {
        $_SESSION['flash_error'] = 'Tidak ada data nilai SAS yang dikirim!';
        header("Location: nilai_sumatif_detail.php?mapel_id=$mapel_id&kelas_id=$kelas_id&tahun_ajaran_id=$tahun_ajaran_id");
        exit();
    }
}

// Get data
$mapel_data = $guruModel->getMapelById($mapel_id);
$kelas_data = $guruModel->getKelasById($kelas_id);
$tahun_ajaran_data = $guruModel->getTahunAjaranById($tahun_ajaran_id);

// Get semua TP untuk mapel dan kelas ini
$tp_list = $guruModel->getTPByMapel($mapel_id, $kelas_id);

// Get siswa di kelas ini
$siswa_list = $guruModel->getSiswaByKelasForNilai($kelas_id);

// Get nilai sumatif akhir (SAS)
$nilai_sas_list = $guruModel->getNilaiSumatifAkhir($mapel_id, $kelas_id);

// Organize data SAS untuk mudah diakses
$nilai_sas_map = [];
foreach ($nilai_sas_list as $nilai_sas) {
    $nilai_sas_map[$nilai_sas['siswa_id']] = $nilai_sas['nilai'];
}

// Get nilai sumatif per TP per siswa
$nilai_tp_map = [];
foreach ($tp_list as $tp) {
    $nilai_tp = $guruModel->getNilaiSumatifTP($tp['id']);
    foreach ($nilai_tp as $nilai) {
        $nilai_tp_map[$nilai['siswa_id']][$tp['id']] = [
            'nilai' => $nilai['nilai'],
            'kode_tp' => $tp['kode_tp']
        ];
    }
}

// ==================== ANALISIS NILAI AKHIR & DESKRIPSI TP ====================

// Konfigurasi presentase dari URL atau default
$presentase_sas = $_GET['presentase_sas'] ?? 60;
$presentase_tp = $_GET['presentase_tp'] ?? 40;

// Validasi presentase
if (($presentase_sas + $presentase_tp) != 100) {
    $presentase_sas = 60;
    $presentase_tp = 40;
}

// Hitung nilai akhir dan analisis TP untuk setiap siswa
$analisis_siswa = [];
$warning_messages = [];

// Debug informasi
echo "<!-- Total siswa: " . count($siswa_list) . " -->\n";
echo "<!-- Total TP: " . count($tp_list) . " -->\n";
echo "<!-- Presentase SAS: $presentase_sas, TP: $presentase_tp -->\n";

foreach ($siswa_list as $siswa) {
    $siswa_id = $siswa['id'];
    
    // Data untuk analisis
    $analisis_data = [
        'siswa_id' => $siswa_id,
        'nama_siswa' => $siswa['nama'],
        'nis' => $siswa['nis'],
        'nilai_sas' => $nilai_sas_map[$siswa_id] ?? null,
        'nilai_tp_list' => [],
        'rata_rata_tp' => 0,
        'nilai_akhir' => 0,
        'tp_tertinggi' => null,
        'tp_terendah' => null,
        'deskripsi_tertinggi' => '',
        'deskripsi_terendah' => '',
        'warning' => ''
    ];
    
    // Debug per siswa
    echo "<!-- Analisis siswa: {$siswa['nama']} (ID: $siswa_id) -->\n";
    echo "<!-- Nilai SAS: " . ($analisis_data['nilai_sas'] ?? 'null') . " -->\n";
    
    // Kumpulkan nilai TP untuk siswa ini
    $nilai_tp_siswa = [];
    foreach ($tp_list as $tp) {
        $nilai_tp = $nilai_tp_map[$siswa_id][$tp['id']]['nilai'] ?? null;
        if ($nilai_tp !== null) {
            $nilai_tp_siswa[] = [
                'tp_id' => $tp['id'],
                'kode_tp' => $tp['kode_tp'],
                'deskripsi' => $tp['deskripsi'],
                'nilai' => $nilai_tp
            ];
        }
    }
    
    $analisis_data['nilai_tp_list'] = $nilai_tp_siswa;
    
    // Debug nilai TP
    echo "<!-- Jumlah TP yang dinilai: " . count($nilai_tp_siswa) . " -->\n";
    foreach ($nilai_tp_siswa as $tp_nilai) {
        echo "<!-- TP {$tp_nilai['kode_tp']}: {$tp_nilai['nilai']} -->\n";
    }
    
    // Hitung rata-rata TP
    if (!empty($nilai_tp_siswa)) {
        $total_nilai_tp = 0;
        foreach ($nilai_tp_siswa as $tp_nilai) {
            $total_nilai_tp += $tp_nilai['nilai'];
        }
        $analisis_data['rata_rata_tp'] = $total_nilai_tp / count($nilai_tp_siswa);
        echo "<!-- Rata-rata TP: {$analisis_data['rata_rata_tp']} -->\n";
    }
    
    // Hitung nilai akhir
    if ($analisis_data['nilai_sas'] !== null && !empty($nilai_tp_siswa)) {
        $analisis_data['nilai_akhir'] = 
            ($analisis_data['nilai_sas'] * $presentase_sas / 100) + 
            ($analisis_data['rata_rata_tp'] * $presentase_tp / 100);
        echo "<!-- Nilai akhir: {$analisis_data['nilai_akhir']} -->\n";
    }
    
    // Analisis TP tertinggi dan terendah
    if (count($nilai_tp_siswa) >= 2) {
        // Urutkan berdasarkan nilai
        usort($nilai_tp_siswa, function($a, $b) {
            return $b['nilai'] - $a['nilai'];
        });
        
        $tp_tertinggi = $nilai_tp_siswa[0];
        $tp_terendah = $nilai_tp_siswa[count($nilai_tp_siswa) - 1];
        
        // Cek apakah ada nilai yang sama
        $nilai_unik = array_unique(array_column($nilai_tp_siswa, 'nilai'));
        
        if (count($nilai_unik) >= 2) {
            $analisis_data['tp_tertinggi'] = $tp_tertinggi;
            $analisis_data['tp_terendah'] = $tp_terendah;
            
            $analisis_data['deskripsi_tertinggi'] = "Menunjukkan penguasaan sangat baik dalam " . $tp_tertinggi['kode_tp'];
            $analisis_data['deskripsi_terendah'] = "Perlu peningkatan dalam " . $tp_terendah['kode_tp'];
            
            echo "<!-- TP Tertinggi: {$tp_tertinggi['kode_tp']} ({$tp_tertinggi['nilai']}) -->\n";
            echo "<!-- TP Terendah: {$tp_terendah['kode_tp']} ({$tp_terendah['nilai']}) -->\n";
        } else {
            $analisis_data['warning'] = "Semua TP memiliki nilai yang sama";
            echo "<!-- Warning: Semua TP memiliki nilai yang sama -->\n";
        }
    } else if (count($nilai_tp_siswa) == 1) {
        $analisis_data['warning'] = "Hanya ada 1 TP yang dinilai - tidak dapat menganalisis";
        echo "<!-- Warning: Hanya 1 TP yang dinilai -->\n";
    } else if (count($nilai_tp_siswa) == 0) {
        $analisis_data['warning'] = "Belum ada TP yang dinilai";
        echo "<!-- Warning: Belum ada TP yang dinilai -->\n";
    }
    
    $analisis_siswa[$siswa_id] = $analisis_data;
}

// Kumpulkan warning messages
foreach ($analisis_siswa as $siswa_analisis) {
    if (!empty($siswa_analisis['warning'])) {
        $warning_messages[] = $siswa_analisis['nama_siswa'] . ": " . $siswa_analisis['warning'];
    }
}

// Debug final
echo "<!-- Total analisis siswa: " . count($analisis_siswa) . " -->\n";
echo "<!-- Total warning: " . count($warning_messages) . " -->\n";
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rekap Nilai Sumatif - Sistem Penilaian</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .navbar-brand {
            font-weight: 600;
        }
        
        .card {
            border-radius: 12px;
            border: none;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .card-header {
            border-radius: 12px 12px 0 0 !important;
            border-bottom: 1px solid rgba(0,0,0,0.05);
            font-weight: 600;
        }
        
        .btn {
            border-radius: 8px;
            font-weight: 500;
        }
        
        .table {
            border-radius: 10px;
            overflow: hidden;
        }
        
        .table thead th {
            background-color: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
            font-weight: 600;
            vertical-align: middle;
            text-align: center;
        }
        
        .table tbody td {
            vertical-align: middle;
        }
        
        .nilai-input {
            max-width: 80px;
            text-align: center;
        }
        
        .hover-row:hover {
            background-color: rgba(0, 123, 255, 0.05) !important;
        }
        
        .stats-card {
            border-left: 4px solid #007bff;
        }
        
        .stats-card.success {
            border-left-color: #28a745;
        }
        
        .stats-card.warning {
            border-left-color: #ffc107;
        }
        
        .stats-card.info {
            border-left-color: #17a2b8;
        }
        
        .fixed-header {
            position: sticky;
            top: 0;
            background: white;
            z-index: 100;
        }
        
        .sticky-column {
            position: sticky;
            left: 0;
            background: white;
            z-index: 10;
            box-shadow: 2px 0 5px rgba(0,0,0,0.1);
        }
        
        .table-responsive {
            max-height: 70vh;
            overflow: auto;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <!-- Header -->
        <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
            <div class="container">
                <a class="navbar-brand" href="#">
                    <i class="bi bi-table"></i> Rekap Nilai Sumatif
                </a>
                <div class="navbar-nav ms-auto">
                    <span class="navbar-text me-3">
                        <i class="bi bi-person-circle"></i> <?= htmlspecialchars($userData['nama']) ?>
                    </span>
					<a href="rekap_nilai.php" class="btn btn-outline-light btn-sm me-2">
						<i class="bi bi-arrow-left"></i> Kembali ke Rekap Nilai
					</a>
                    <a href="dashboard.php" class="btn btn-outline-light btn-sm">
                        <i class="bi bi-house"></i> Dashboard
                    </a>
                </div>
            </div>
        </nav>

        <div class="container mt-4">
            <!-- Breadcrumb -->
			<!-- Breadcrumb -->
			<nav aria-label="breadcrumb" class="mb-4">
				<ol class="breadcrumb">
					<li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
					<li class="breadcrumb-item"><a href="rekap_nilai.php">Rekap Nilai</a></li>
					<li class="breadcrumb-item active">Detail Nilai Sumatif</li>
				</ol>
			</nav>

            <!-- Flash Messages -->
            <?php if (isset($_SESSION['flash_success'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="bi bi-check-circle-fill"></i> <?= $_SESSION['flash_success'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php unset($_SESSION['flash_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['flash_error'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bi bi-exclamation-triangle-fill"></i> <?= $_SESSION['flash_error'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php unset($_SESSION['flash_error']); ?>
            <?php endif; ?>

<!-- Warning Messages untuk Analisis TP -->
<?php if (!empty($warning_messages)): ?>
    <div class="alert alert-warning alert-dismissible fade show" role="alert">
        <h6><i class="bi bi-exclamation-triangle"></i> Peringatan Analisis TP</h6>
        <small>
            <?php foreach (array_slice($warning_messages, 0, 5) as $warning): ?>
                • <?= htmlspecialchars($warning) ?><br>
            <?php endforeach; ?>
            <?php if (count($warning_messages) > 5): ?>
                • ... dan <?= count($warning_messages) - 5 ?> warning lainnya
            <?php endif; ?>
        </small>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

            <!-- Header Info -->
            <div class="card mb-4">
                <div class="card-header bg-primary text-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h5 class="mb-0"><?= htmlspecialchars($mapel_data['nama_mapel'] ?? 'Mata Pelajaran') ?></h5>
                            <small>
                                Kelas: <?= htmlspecialchars($kelas_data['nama_kelas'] ?? 'Kelas') ?> - 
                                Tahun Ajaran: <?= htmlspecialchars($tahun_ajaran_data['tahun_ajaran'] ?? 'Tahun Ajaran') ?> 
                                (<?= htmlspecialchars($tahun_ajaran_data['semester'] ?? 'Semester') ?>)
                            </small>
                        </div>
                        <div class="text-end">
                            <small>
                                Total TP: <?= count($tp_list) ?> | 
                                Total Siswa: <?= count($siswa_list) ?>
                            </small>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <div class="row text-center">
                        <div class="col-md-3">
                            <div class="card stats-card">
                                <div class="card-body py-3">
                                    <div class="text-muted small">Total Siswa</div>
                                    <div class="h5 mb-0 text-primary"><?= count($siswa_list) ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card stats-card success">
                                <div class="card-body py-3">
                                    <div class="text-muted small">Total TP</div>
                                    <div class="h5 mb-0 text-success"><?= count($tp_list) ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card stats-card warning">
                                <div class="card-body py-3">
                                    <div class="text-muted small">Siswa dengan SAS</div>
                                    <div class="h5 mb-0 text-warning"><?= count($nilai_sas_map) ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card stats-card info">
                                <div class="card-body py-3">
                                    <div class="text-muted small">Rata-rata SAS</div>
                                    <div class="h5 mb-0 text-info">
                                        <?= count($nilai_sas_map) > 0 ? number_format(array_sum($nilai_sas_map) / count($nilai_sas_map), 2) : '0.00' ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Rekap Nilai Table -->
            <div class="card">
			
			
				<div class="card-header bg-white py-3 d-flex justify-content-between align-items-center">
					<h6 class="m-0 font-weight-bold text-primary">
						<i class="bi bi-table"></i> Rekap Nilai Sumatif
					</h6>
					<div>
						<!-- Tombol Export Excel -->
						<a href="export_nilai_sumatif.php?mapel_id=<?= $mapel_id ?>&kelas_id=<?= $kelas_id ?>&tahun_ajaran_id=<?= $tahun_ajaran_id ?>&presentase_sas=<?= $presentase_sas ?>&presentase_tp=<?= $presentase_tp ?>" 
						   class="btn btn-success btn-sm me-2" 
						   title="Export ke Excel">
							<i class="bi bi-file-earmark-excel"></i> Export Excel
						</a>
						<button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#saveSASModal">
							<i class="bi bi-save"></i> Simpan SAS
						</button>
					</div>
				</div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <form id="nilaiForm" method="POST">
                            <input type="hidden" name="update_sas" value="1">
                            
                            <table class="table table-bordered table-hover mb-0">
                                <thead class="table-light fixed-header">
									<tr>
										<th class="sticky-column" width="200">Nama Siswa</th>
										<th class="sticky-column" width="100">NIS</th>
										<?php foreach ($tp_list as $tp): ?>
											<th width="120" title="<?= htmlspecialchars($tp['deskripsi']) ?>">
												<?= htmlspecialchars($tp['kode_tp']) ?>
											</th>
										<?php endforeach; ?>
										<th width="120" class="bg-warning text-dark">
											<strong>SAS</strong>
										</th>
										<!-- Kolom Baru -->
										<th width="120" class="bg-success text-white">
											<strong>Nilai Akhir</strong>
											<br>
											<small>(<?= $presentase_sas ?>:<?= $presentase_tp ?>)</small>
										</th>
										<th width="200" class="bg-info text-white">
											<strong>Deskripsi TP Tertinggi</strong>
										</th>
										<th width="200" class="bg-danger text-white">
											<strong>Deskripsi TP Terendah</strong>
										</th>
									</tr>
								</thead>
								
								<tbody>
    <?php if (!empty($siswa_list)): ?>
        <?php foreach ($siswa_list as $siswa): ?>
            <?php $analisis = $analisis_siswa[$siswa['id']] ?? []; ?>
            <tr class="hover-row">
                <td class="sticky-column bg-light">
                    <strong><?= htmlspecialchars($siswa['nama']) ?></strong>
                </td>
                <td class="sticky-column bg-light">
                    <?= htmlspecialchars($siswa['nis']) ?>
                </td>
                
                <?php foreach ($tp_list as $tp): ?>
                    <?php 
                    $nilai_tp = $nilai_tp_map[$siswa['id']][$tp['id']]['nilai'] ?? '';
                    $badge_class = $nilai_tp ? 'bg-success' : 'bg-secondary';
                    ?>
                    <td class="text-center">
                        <?php if ($nilai_tp): ?>
                            <span class="badge <?= $badge_class ?>">
                                <?= number_format($nilai_tp, 2) ?>
                            </span>
                        <?php else: ?>
                            <span class="badge bg-light text-muted">-</span>
                        <?php endif; ?>
                    </td>
                <?php endforeach; ?>
                
                <td class="text-center bg-warning bg-opacity-10">
                    <input type="number" 
                           class="form-control form-control-sm nilai-input" 
                           name="nilai_sas[<?= $siswa['id'] ?>]" 
                           value="<?= $nilai_sas_map[$siswa['id']] ?? '' ?>" 
                           min="0" max="100" step="0.01" 
                           placeholder="0-100"
                           style="font-weight: bold;">
                </td>
                
                <!-- Kolom Nilai Akhir -->
                <td class="text-center bg-success bg-opacity-10">
                    <?php if ($analisis['nilai_akhir'] > 0): ?>
                        <span class="badge bg-success fs-6">
                            <?= number_format($analisis['nilai_akhir'], 2) ?>
                        </span>
                        <br>
                        <small class="text-muted">
                            SAS: <?= number_format($analisis['nilai_sas'], 1) ?><br>
                            TP: <?= number_format($analisis['rata_rata_tp'], 1) ?>
                        </small>
                    <?php else: ?>
                        <span class="badge bg-secondary">-</span>
                    <?php endif; ?>
                </td>
                
                <!-- Kolom Deskripsi TP Tertinggi -->
                <td class="bg-info bg-opacity-10">
                    <?php if (!empty($analisis['deskripsi_tertinggi'])): ?>
                        <small><?= htmlspecialchars($analisis['deskripsi_tertinggi']) ?></small>
                        <br>
                        <span class="badge bg-success">
                            <?= number_format($analisis['tp_tertinggi']['nilai'], 1) ?>
                        </span>
                    <?php elseif (!empty($analisis['warning'])): ?>
                        <small class="text-warning"><?= htmlspecialchars($analisis['warning']) ?></small>
                    <?php else: ?>
                        <small class="text-muted">-</small>
                    <?php endif; ?>
                </td>
                
                <!-- Kolom Deskripsi TP Terendah -->
                <td class="bg-danger bg-opacity-10">
                    <?php if (!empty($analisis['deskripsi_terendah'])): ?>
                        <small><?= htmlspecialchars($analisis['deskripsi_terendah']) ?></small>
                        <br>
                        <span class="badge bg-danger">
                            <?= number_format($analisis['tp_terendah']['nilai'], 1) ?>
                        </span>
                    <?php elseif (!empty($analisis['warning'])): ?>
                        <small class="text-warning"><?= htmlspecialchars($analisis['warning']) ?></small>
                    <?php else: ?>
                        <small class="text-muted">-</small>
                    <?php endif; ?>
                </td>
            </tr>
        <?php endforeach; ?>
    <?php else: ?>
        <tr>
            <td colspan="<?= count($tp_list) + 5 ?>" class="text-center py-4">
                <i class="bi bi-people fs-1 text-muted"></i>
                <p class="text-muted mt-2">Tidak ada siswa dalam kelas ini.</p>
            </td>
        </tr>
    <?php endif; ?>
</tbody>
								
                            </table>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Legend -->
            <div class="card mt-4">
                <div class="card-body">
                    <h6 class="mb-3"><i class="bi bi-info-circle"></i> Keterangan:</h6>
                    <div class="row">
                        <div class="col-md-4">
                            <span class="badge bg-success me-2">✓</span> Nilai TP sudah diinput
                        </div>
                        <div class="col-md-4">
                            <span class="badge bg-secondary me-2">-</span> Nilai TP belum diinput
                        </div>
                        <div class="col-md-4">
                            <span class="badge bg-warning me-2">SAS</span> Nilai Sumatif Akhir Semester
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Save SAS Modal -->
	<!-- Save SAS Modal -->
<div class="modal fade" id="saveSASModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bi bi-save text-success"></i> Simpan Nilai SAS
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Anda akan menyimpan nilai Sumatif Akhir Semester (SAS) untuk semua siswa.</p>
                <div class="alert alert-info">
                    <small>
                        <i class="bi bi-info-circle"></i> 
                        <strong>Informasi:</strong> Hanya nilai SAS yang sudah diisi dan valid (0-100) yang akan disimpan.
                    </small>
                </div>
                <div class="alert alert-warning">
                    <small>
                        <i class="bi bi-exclamation-triangle"></i> 
                        <strong>Perhatian:</strong> Pastikan nilai SAS sudah benar sebelum disimpan.
                    </small>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="bi bi-x-circle"></i> Batal
                </button>
                <button type="submit" form="nilaiForm" class="btn btn-success" id="confirmSaveSAS">
                    <i class="bi bi-save"></i> Simpan SAS
                </button>
            </div>
        </div>
    </div>
</div>
	
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
    // Auto-hide alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            if (alert) {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            }
        }, 5000);
    });

    // Highlight changed SAS values
    const sasInputs = document.querySelectorAll('input[name^="nilai_sas"]');
    const initialValues = {};
    
    sasInputs.forEach(input => {
        initialValues[input.name] = input.value;
        
        input.addEventListener('change', function() {
            if (this.value !== initialValues[this.name]) {
                this.style.backgroundColor = '#fff3cd';
                this.style.borderColor = '#ffc107';
            } else {
                this.style.backgroundColor = '';
                this.style.borderColor = '';
            }
        });

        // Real-time validation
        input.addEventListener('blur', function() {
            const value = parseFloat(this.value);
            if (this.value !== '' && (isNaN(value) || value < 0 || value > 100)) {
                this.style.backgroundColor = '#f8d7da';
                this.style.borderColor = '#dc3545';
            } else if (this.value !== '') {
                this.style.backgroundColor = '#d1edff';
                this.style.borderColor = '#0d6efd';
            }
        });
    });

    // Form validation
    const form = document.getElementById('nilaiForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            let hasValidSASValue = false;
            let invalidValues = [];
            
            sasInputs.forEach(input => {
                const value = input.value.trim();
                if (value !== '') {
                    const numValue = parseFloat(value);
                    if (!isNaN(numValue) && numValue >= 0 && numValue <= 100) {
                        hasValidSASValue = true;
                    } else {
                        invalidValues.push(input.closest('tr').querySelector('td:first-child').textContent.trim());
                    }
                }
            });
            
            if (!hasValidSASValue) {
                e.preventDefault();
                alert('Masukkan setidaknya satu nilai SAS yang valid (0-100) sebelum menyimpan!');
                return false;
            }
            
            if (invalidValues.length > 0) {
                e.preventDefault();
                alert('Beberapa nilai SAS tidak valid untuk siswa:\n' + invalidValues.join('\n') + '\n\nNilai harus antara 0-100.');
                return false;
            }

            // Show loading state
            const submitBtn = document.querySelector('button[type="submit"]');
            if (submitBtn) {
                submitBtn.innerHTML = '<i class="bi bi-hourglass-split"></i> Menyimpan...';
                submitBtn.disabled = true;
            }
        });
    }

    // Tooltip for TP descriptions
    const tpHeaders = document.querySelectorAll('th[title]');
    tpHeaders.forEach(header => {
        new bootstrap.Tooltip(header);
    });

    // Auto-save on Enter key in SAS inputs
    sasInputs.forEach(input => {
        input.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                document.getElementById('saveSASModal').style.display = 'block';
                new bootstrap.Modal(document.getElementById('saveSASModal')).show();
            }
        });
    });
});
    </script>
</body>
</html>