<?php
// Enable error reporting untuk debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../../app/config/config.php';

// Check if PhpSpreadsheet is available
if (!class_exists('PhpOffice\PhpSpreadsheet\Spreadsheet')) {
    die("Error: PHP Spreadsheet library tidak ditemukan. Silakan install via Composer: composer require phpoffice/phpspreadsheet");
}

Session::checkAuth();
Session::checkRole(['guru']);

try {
    use PhpOffice\PhpSpreadsheet\Spreadsheet;
    use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
    use PhpOffice\PhpSpreadsheet\Style\Border;
    use PhpOffice\PhpSpreadsheet\Style\Alignment;
    use PhpOffice\PhpSpreadsheet\Style\Fill;

    $guruModel = new GuruModel();
    $userData = Session::getUserData();
    $guruId = $userData['guru_id'];

    // Get parameters
    $mapel_id = $_GET['mapel_id'] ?? null;
    $kelas_id = $_GET['kelas_id'] ?? null;
    $tahun_ajaran_id = $_GET['tahun_ajaran_id'] ?? null;
    $presentase_sas = $_GET['presentase_sas'] ?? 60;
    $presentase_tp = $_GET['presentase_tp'] ?? 40;

    // Validasi parameter
    if (!$mapel_id || !$kelas_id) {
        throw new Exception("Parameter tidak lengkap: mapel_id dan kelas_id diperlukan");
    }

    // Validasi akses guru
    if (!$guruModel->validateGuruMapelAccess($guruId, $mapel_id, $kelas_id)) {
        throw new Exception("Akses ditolak untuk mapel dan kelas ini.");
    }

    // Get data
    $mapel_data = $guruModel->getMapelById($mapel_id);
    $kelas_data = $guruModel->getKelasById($kelas_id);
    $tahun_ajaran_data = $guruModel->getTahunAjaranById($tahun_ajaran_id);
    $tp_list = $guruModel->getTPByMapel($mapel_id, $kelas_id);
    $siswa_list = $guruModel->getSiswaByKelasForNilai($kelas_id);
    $nilai_sas_list = $guruModel->getNilaiSumatifAkhir($mapel_id, $kelas_id);

    // Validasi data
    if (empty($siswa_list)) {
        throw new Exception("Tidak ada siswa dalam kelas ini.");
    }

    if (empty($tp_list)) {
        throw new Exception("Tidak ada Tujuan Pembelajaran untuk mapel dan kelas ini.");
    }

    // Organize data
    $nilai_sas_map = [];
    foreach ($nilai_sas_list as $nilai_sas) {
        $nilai_sas_map[$nilai_sas['siswa_id']] = $nilai_sas['nilai'];
    }

    $nilai_tp_map = [];
    foreach ($tp_list as $tp) {
        $nilai_tp = $guruModel->getNilaiSumatifTP($tp['id']);
        foreach ($nilai_tp as $nilai) {
            $nilai_tp_map[$nilai['siswa_id']][$tp['id']] = [
                'nilai' => $nilai['nilai'],
                'kode_tp' => $tp['kode_tp']
            ];
        }
    }

    // Hitung analisis sama seperti di halaman utama
    $analisis_siswa = [];
    foreach ($siswa_list as $siswa) {
        $siswa_id = $siswa['id'];
        
        $analisis_data = [
            'siswa_id' => $siswa_id,
            'nama_siswa' => $siswa['nama'],
            'nis' => $siswa['nis'],
            'nilai_sas' => $nilai_sas_map[$siswa_id] ?? null,
            'nilai_tp_list' => [],
            'rata_rata_tp' => 0,
            'nilai_akhir' => 0,
            'tp_tertinggi' => null,
            'tp_terendah' => null,
            'deskripsi_tertinggi' => '',
            'deskripsi_terendah' => ''
        ];
        
        // Kumpulkan nilai TP
        $nilai_tp_siswa = [];
        foreach ($tp_list as $tp) {
            $nilai_tp = $nilai_tp_map[$siswa_id][$tp['id']]['nilai'] ?? null;
            if ($nilai_tp !== null) {
                $nilai_tp_siswa[] = [
                    'tp_id' => $tp['id'],
                    'kode_tp' => $tp['kode_tp'],
                    'deskripsi' => $tp['deskripsi'],
                    'nilai' => $nilai_tp
                ];
            }
        }
        
        $analisis_data['nilai_tp_list'] = $nilai_tp_siswa;
        
        // Hitung rata-rata TP
        if (!empty($nilai_tp_siswa)) {
            $total_nilai_tp = 0;
            foreach ($nilai_tp_siswa as $tp_nilai) {
                $total_nilai_tp += $tp_nilai['nilai'];
            }
            $analisis_data['rata_rata_tp'] = $total_nilai_tp / count($nilai_tp_siswa);
        }
        
        // Hitung nilai akhir
        if ($analisis_data['nilai_sas'] !== null && !empty($nilai_tp_siswa)) {
            $analisis_data['nilai_akhir'] = 
                ($analisis_data['nilai_sas'] * $presentase_sas / 100) + 
                ($analisis_data['rata_rata_tp'] * $presentase_tp / 100);
        }
        
        // Analisis TP tertinggi dan terendah
        if (count($nilai_tp_siswa) >= 2) {
            usort($nilai_tp_siswa, function($a, $b) {
                return $b['nilai'] - $a['nilai'];
            });
            
            $tp_tertinggi = $nilai_tp_siswa[0];
            $tp_terendah = $nilai_tp_siswa[count($nilai_tp_siswa) - 1];
            
            $nilai_unik = array_unique(array_column($nilai_tp_siswa, 'nilai'));
            
            if (count($nilai_unik) >= 2) {
                $analisis_data['tp_tertinggi'] = $tp_tertinggi;
                $analisis_data['tp_terendah'] = $tp_terendah;
                $analisis_data['deskripsi_tertinggi'] = "Menunjukkan penguasaan sangat baik dalam " . $tp_tertinggi['kode_tp'];
                $analisis_data['deskripsi_terendah'] = "Perlu peningkatan dalam " . $tp_terendah['kode_tp'];
            }
        }
        
        $analisis_siswa[$siswa_id] = $analisis_data;
    }

    // Create Spreadsheet
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();

    // Set judul
    $sheet->setTitle('Rekap Nilai Sumatif');

    // Header informasi
    $sheet->setCellValue('A1', 'REKAP NILAI SUMATIF');
    $sheet->mergeCells('A1:G1');
    $sheet->setCellValue('A2', 'Mata Pelajaran: ' . ($mapel_data['nama_mapel'] ?? ''));
    $sheet->mergeCells('A2:G2');
    $sheet->setCellValue('A3', 'Kelas: ' . ($kelas_data['nama_kelas'] ?? ''));
    $sheet->mergeCells('A3:G3');
    $sheet->setCellValue('A4', 'Tahun Ajaran: ' . ($tahun_ajaran_data['tahun_ajaran'] ?? '') . ' - Semester: ' . ($tahun_ajaran_data['semester'] ?? ''));
    $sheet->mergeCells('A4:G4');
    $sheet->setCellValue('A5', 'Konfigurasi Nilai: SAS ' . $presentase_sas . '% + Rata-rata TP ' . $presentase_tp . '%');
    $sheet->mergeCells('A5:G5');
    $sheet->setCellValue('A6', 'Tanggal Export: ' . date('d/m/Y H:i:s'));
    $sheet->mergeCells('A6:G6');

    // Header tabel
    $headers = ['No', 'NIS', 'Nama Siswa'];
    foreach ($tp_list as $tp) {
        $headers[] = $tp['kode_tp'];
    }
    $headers = array_merge($headers, [
        'SAS', 
        'Rata-rata TP', 
        'Nilai Akhir',
        'TP Tertinggi (Nilai)',
        'TP Terendah (Nilai)',
        'Deskripsi TP Tertinggi',
        'Deskripsi TP Terendah'
    ]);

    // Set header
    $col = 1;
    foreach ($headers as $header) {
        $sheet->setCellValueByColumnAndRow($col, 8, $header);
        $col++;
    }

    // Style untuk header
    $headerStyle = [
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '2E86C1']],
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]]
    ];

    $lastHeaderCol = $sheet->getColumnStringFromColumnIndex(count($headers));
    $sheet->getStyle('A8:' . $lastHeaderCol . '8')->applyFromArray($headerStyle);

    // Isi data
    $row = 9;
    $no = 1;
    foreach ($siswa_list as $siswa) {
        $analisis = $analisis_siswa[$siswa['id']] ?? [];
        
        $sheet->setCellValue('A' . $row, $no);
        $sheet->setCellValue('B' . $row, $siswa['nis']);
        $sheet->setCellValue('C' . $row, $siswa['nama']);
        
        // Nilai TP
        $col = 4; // Mulai dari kolom D
        foreach ($tp_list as $tp) {
            $nilai_tp = $nilai_tp_map[$siswa['id']][$tp['id']]['nilai'] ?? '';
            $sheet->setCellValueByColumnAndRow($col, $row, $nilai_tp !== '' ? $nilai_tp : '-');
            $col++;
        }
        
        // SAS
        $sheet->setCellValueByColumnAndRow($col, $row, $analisis['nilai_sas'] ?? '');
        $col++;
        
        // Rata-rata TP
        $sheet->setCellValueByColumnAndRow($col, $row, !empty($analisis['rata_rata_tp']) ? round($analisis['rata_rata_tp'], 2) : '');
        $col++;
        
        // Nilai Akhir
        $sheet->setCellValueByColumnAndRow($col, $row, !empty($analisis['nilai_akhir']) ? round($analisis['nilai_akhir'], 2) : '');
        $col++;
        
        // TP Tertinggi
        $tp_tertinggi = $analisis['tp_tertinggi'] ?? null;
        $sheet->setCellValueByColumnAndRow($col, $row, $tp_tertinggi ? $tp_tertinggi['kode_tp'] . ' (' . $tp_tertinggi['nilai'] . ')' : '');
        $col++;
        
        // TP Terendah
        $tp_terendah = $analisis['tp_terendah'] ?? null;
        $sheet->setCellValueByColumnAndRow($col, $row, $tp_terendah ? $tp_terendah['kode_tp'] . ' (' . $tp_terendah['nilai'] . ')' : '');
        $col++;
        
        // Deskripsi TP Tertinggi
        $sheet->setCellValueByColumnAndRow($col, $row, $analisis['deskripsi_tertinggi'] ?? '');
        $col++;
        
        // Deskripsi TP Terendah
        $sheet->setCellValueByColumnAndRow($col, $row, $analisis['deskripsi_terendah'] ?? '');
        
        $row++;
        $no++;
    }

    // Auto size columns
    foreach (range('A', $lastHeaderCol) as $column) {
        $sheet->getColumnDimension($column)->setAutoSize(true);
    }

    // Style untuk judul
    $titleStyle = [
        'font' => ['bold' => true, 'size' => 14],
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
    ];
    $sheet->getStyle('A1:A6')->applyFromArray($titleStyle);

    // Style untuk data
    $dataStyle = [
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]],
        'alignment' => ['vertical' => Alignment::VERTICAL_CENTER]
    ];
    $sheet->getStyle('A8:' . $lastHeaderCol . ($row-1))->applyFromArray($dataStyle);

    // Style untuk angka (rata-rata kanan)
    $numberLastCol = $sheet->getColumnStringFromColumnIndex(3 + count($tp_list) + 2);
    $numberStyle = [
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]
    ];
    $sheet->getStyle('D9:' . $numberLastCol . ($row-1))->applyFromArray($numberStyle);

    // Set header untuk download
    $filename = 'Rekap_Nilai_Sumatif_' . 
               ($mapel_data['nama_mapel'] ?? 'Mapel') . '_' . 
               ($kelas_data['nama_kelas'] ?? 'Kelas') . '_' . 
               date('Y-m-d') . '.xlsx';

    // Clear any previous output
    if (ob_get_length()) {
        ob_end_clean();
    }

    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    header('Cache-Control: max-age=1');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
    header('Cache-Control: cache, must-revalidate');
    header('Pragma: public');

    $writer = new Xlsx($spreadsheet);
    $writer->save('php://output');
    exit;

} catch (Exception $e) {
    // Log error
    error_log("Export Error: " . $e->getMessage());
    
    // Tampilkan error untuk debugging
    if (ini_get('display_errors')) {
        die("Error: " . $e->getMessage() . " pada baris " . $e->getLine());
    } else {
        die("Terjadi kesalahan saat mengexport data. Silakan coba lagi.");
    }
}