// guru-nilai.js - Modular JavaScript for Guru Nilai Management
class NilaiManager {
    constructor() {
        this.initEventListeners();
        this.initAutoSave();
    }

    initEventListeners() {
        // Edit nilai buttons
        document.querySelectorAll('.edit-nilai').forEach(btn => {
            btn.addEventListener('click', (e) => this.handleEditNilai(e));
        });

        // Form validation
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', (e) => this.validateNilaiForm(e));
        });

        // Real-time calculation
        document.querySelectorAll('input[name="nilai"]').forEach(input => {
            input.addEventListener('input', (e) => this.calculateStatistics(e));
        });
    }

    initAutoSave() {
        // Auto-save draft every 30 seconds
        setInterval(() => {
            this.saveDraft();
        }, 30000);

        // Save on page unload
        window.addEventListener('beforeunload', (e) => {
            if (this.hasUnsavedChanges()) {
                this.saveDraft();
            }
        });
    }

    handleEditNilai(event) {
        const button = event.currentTarget;
        const nilaiId = button.dataset.nilaiId;
        const jenis = button.dataset.jenis;
        const nilai = button.dataset.nilai;
        const deskripsi = button.dataset.deskripsi;

        // Create edit form modal or inline edit
        this.showEditForm(nilaiId, jenis, nilai, deskripsi);
    }

    showEditForm(nilaiId, jenis, nilai, deskripsi) {
        const modalHtml = `
            <div class="modal fade" id="editNilaiModal" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Edit Nilai</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="editNilaiForm">
                                <input type="hidden" name="nilai_id" value="${nilaiId}">
                                <div class="mb-3">
                                    <label class="form-label">Jenis Penilaian</label>
                                    <input type="text" class="form-control" value="${this.formatJenis(jenis)}" readonly>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Nilai</label>
                                    <input type="number" class="form-control" name="nilai" 
                                           value="${nilai}" min="0" max="100" step="0.01" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Deskripsi</label>
                                    <textarea class="form-control" name="deskripsi" rows="3">${deskripsi}</textarea>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                            <button type="button" class="btn btn-primary" onclick="nilaiManager.updateNilai()">Simpan</button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal if any
        const existingModal = document.getElementById('editNilaiModal');
        if (existingModal) {
            existingModal.remove();
        }

        document.body.insertAdjacentHTML('beforeend', modalHtml);
        const modal = new bootstrap.Modal(document.getElementById('editNilaiModal'));
        modal.show();
    }

    async updateNilai() {
        const form = document.getElementById('editNilaiForm');
        const formData = new FormData(form);

        try {
            const response = await fetch('../../app/ajax/nilai_ajax.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.showAlert('Nilai berhasil diperbarui', 'success');
                bootstrap.Modal.getInstance(document.getElementById('editNilaiModal')).hide();
                setTimeout(() => location.reload(), 1500);
            } else {
                this.showAlert(result.message || 'Gagal memperbarui nilai', 'error');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showAlert('Terjadi kesalahan saat memperbarui nilai', 'error');
        }
    }

    validateNilaiForm(event) {
        const form = event.target;
        const nilaiInput = form.querySelector('input[name="nilai"]');
        const jenisSelect = form.querySelector('select[name="jenis"]');

        if (!nilaiInput || !jenisSelect) return;

        const nilai = parseFloat(nilaiInput.value);
        const jenis = jenisSelect.value;

        if (isNaN(nilai) || nilai < 0 || nilai > 100) {
            event.preventDefault();
            this.showAlert('Nilai harus berupa angka antara 0 - 100', 'error');
            return;
        }

        if (!jenis) {
            event.preventDefault();
            this.showAlert('Pilih jenis penilaian', 'error');
            return;
        }

        // Show loading indicator
        const submitBtn = form.querySelector('button[type="submit"]');
        if (submitBtn) {
            submitBtn.innerHTML = '<i class="bi bi-hourglass-split me-1"></i>Menyimpan...';
            submitBtn.disabled = true;
        }
    }

    calculateStatistics(event) {
        const input = event.target;
        const row = input.closest('tr');
        const allNilaiInputs = document.querySelectorAll('input[name="nilai"]');
        
        let total = 0;
        let count = 0;
        let min = 100;
        let max = 0;

        allNilaiInputs.forEach(input => {
            const nilai = parseFloat(input.value);
            if (!isNaN(nilai)) {
                total += nilai;
                count++;
                min = Math.min(min, nilai);
                max = Math.max(max, nilai);
            }
        });

        const average = count > 0 ? total / count : 0;

        // Update statistics display
        this.updateStatisticsDisplay(average, min, max, count);
    }

    updateStatisticsDisplay(average, min, max, count) {
        let statsContainer = document.getElementById('nilaiStatistics');
        
        if (!statsContainer) {
            statsContainer = document.createElement('div');
            statsContainer.id = 'nilaiStatistics';
            statsContainer.className = 'card bg-light mb-3';
            statsContainer.innerHTML = `
                <div class="card-body">
                    <h6 class="card-title">Statistik Nilai</h6>
                    <div class="row text-center">
                        <div class="col-3">
                            <small class="text-muted">Rata-rata</small>
                            <div class="h5 mb-0" id="avgNilai">0</div>
                        </div>
                        <div class="col-3">
                            <small class="text-muted">Terendah</small>
                            <div class="h5 mb-0" id="minNilai">0</div>
                        </div>
                        <div class="col-3">
                            <small class="text-muted">Tertinggi</small>
                            <div class="h5 mb-0" id="maxNilai">0</div>
                        </div>
                        <div class="col-3">
                            <small class="text-muted">Jumlah</small>
                            <div class="h5 mb-0" id="countNilai">0</div>
                        </div>
                    </div>
                </div>
            `;
            
            const main = document.querySelector('main');
            const firstCard = main.querySelector('.card');
            if (firstCard) {
                firstCard.parentNode.insertBefore(statsContainer, firstCard);
            }
        }

        document.getElementById('avgNilai').textContent = average.toFixed(2);
        document.getElementById('minNilai').textContent = min.toFixed(2);
        document.getElementById('maxNilai').textContent = max.toFixed(2);
        document.getElementById('countNilai').textContent = count;
    }

    saveDraft() {
        const forms = document.querySelectorAll('form');
        const draftData = {};

        forms.forEach((form, index) => {
            const formData = new FormData(form);
            draftData[`form_${index}`] = Object.fromEntries(formData);
        });

        localStorage.setItem('nilaiDraft', JSON.stringify(draftData));
        console.log('Draft saved');
    }

    loadDraft() {
        const draftData = localStorage.getItem('nilaiDraft');
        if (draftData) {
            const data = JSON.parse(draftData);
            // Implement draft loading logic
            console.log('Draft loaded:', data);
        }
    }

    hasUnsavedChanges() {
        const forms = document.querySelectorAll('form');
        let hasChanges = false;

        forms.forEach(form => {
            const inputs = form.querySelectorAll('input, select, textarea');
            inputs.forEach(input => {
                if (input.value !== input.defaultValue) {
                    hasChanges = true;
                }
            });
        });

        return hasChanges;
    }

    formatJenis(jenis) {
        const jenisMap = {
            'observasi': 'Observasi',
            'tugas_tertulis': 'Tugas Tertulis',
            'projek': 'Projek',
            'portofolio': 'Portofolio',
            'praktik': 'Praktik',
            'wawancara': 'Wawancara'
        };
        
        return jenisMap[jenis] || jenis;
    }

    showAlert(message, type = 'info') {
        const alertClass = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'info': 'alert-info',
            'warning': 'alert-warning'
        }[type] || 'alert-info';
        
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert ${alertClass} alert-dismissible fade show`;
        alertDiv.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        const main = document.querySelector('main');
        main.insertBefore(alertDiv, main.firstChild);
        
        setTimeout(() => {
            if (alertDiv.parentNode) {
                alertDiv.remove();
            }
        }, 5000);
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.nilaiManager = new NilaiManager();
    nilaiManager.loadDraft();
});