// Edit Siswa functionality - FIXED BACKDROP ISSUE
document.addEventListener('DOMContentLoaded', function() {
    // Edit Siswa
    document.querySelectorAll('.edit-siswa').forEach(btn => {
        btn.addEventListener('click', function() {
            const siswaId = this.getAttribute('data-id');
            loadSiswaData(siswaId);
        });
    });

    // Delete Siswa
    document.querySelectorAll('.delete-siswa').forEach(btn => {
        btn.addEventListener('click', function() {
            const siswaId = this.getAttribute('data-id');
            const siswaNama = this.getAttribute('data-nama');
            
            document.getElementById('delete_siswa_id').value = siswaId;
            document.getElementById('delete_siswa_nama').textContent = siswaNama;
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteSiswaModal'));
            deleteModal.show();
        });
    });

    // Delete confirmation
    const confirmDeleteCheckbox = document.getElementById('confirmDeleteSiswa');
    if (confirmDeleteCheckbox) {
        confirmDeleteCheckbox.addEventListener('change', function() {
            document.getElementById('deleteSiswaConfirmBtn').disabled = !this.checked;
        });
    }

    // FIX: Enhanced backdrop cleanup
    setupBackdropCleanup();
});

function setupBackdropCleanup() {
    // Clean up any existing backdrops on page load
    removeStuckBackdrops();
    
    // Monitor all modals for hidden event
    const modals = ['editSiswaModal', 'deleteSiswaModal', 'addSiswaModal'];
    
    modals.forEach(modalId => {
        const modalElement = document.getElementById(modalId);
        if (modalElement) {
            // FIX: Force cleanup when modal is hidden
            modalElement.addEventListener('hidden.bs.modal', function () {
                forceBackdropCleanup();
                resetModalForm(this);
            });

            // FIX: Additional cleanup on show to prevent stacking
            modalElement.addEventListener('show.bs.modal', function () {
                removeStuckBackdrops();
            });
        }
    });

    // FIX: Special handler for cancel buttons
    document.addEventListener('click', function(e) {
        if (e.target.matches('.btn-secondary[data-bs-dismiss="modal"]') || 
            e.target.closest('.btn-secondary[data-bs-dismiss="modal"]')) {
            
            e.preventDefault();
            e.stopPropagation();
            
            // Find the modal
            const modal = e.target.closest('.modal') || 
                         e.target.closest('[data-bs-dismiss="modal"]')?.closest('.modal');
            
            if (modal) {
                // Use Bootstrap's hide method
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) {
                    bsModal.hide();
                }
                
                // Force cleanup after a short delay
                setTimeout(forceBackdropCleanup, 150);
            }
        }
    });

    // FIX: Handle escape key with cleanup
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                const bsModal = bootstrap.Modal.getInstance(openModal);
                if (bsModal) {
                    bsModal.hide();
                }
                setTimeout(forceBackdropCleanup, 150);
            }
        }
    });

    // FIX: Handle backdrop clicks directly
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal-backdrop')) {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                const bsModal = bootstrap.Modal.getInstance(openModal);
                if (bsModal) {
                    bsModal.hide();
                }
            }
            e.target.remove();
            document.body.classList.remove('modal-open');
        }
    });
}

function forceBackdropCleanup() {
    // Remove all backdrops
    const backdrops = document.querySelectorAll('.modal-backdrop');
    backdrops.forEach(backdrop => {
        backdrop.remove();
    });
    
    // Reset body classes and styles
    document.body.classList.remove('modal-open');
    document.body.style.overflow = '';
    document.body.style.paddingRight = '';
    
    // Hide any still-visible modals
    const visibleModals = document.querySelectorAll('.modal');
    visibleModals.forEach(modal => {
        modal.style.display = 'none';
        modal.classList.remove('show');
    });
}

function removeStuckBackdrops() {
    // Clean up any stuck backdrops on page load or modal show
    const backdrops = document.querySelectorAll('.modal-backdrop');
    if (backdrops.length > 0) {
        forceBackdropCleanup();
    }
}

function resetModalForm(modal) {
    // Reset forms in modal
    const forms = modal.querySelectorAll('form');
    forms.forEach(form => {
        form.reset();
    });

    // Specific reset for edit modal
    if (modal.id === 'editSiswaModal') {
        const loading = document.getElementById('editLoading');
        const formContent = document.getElementById('editFormContent');
        
        if (loading) loading.classList.add('d-none');
        if (formContent) formContent.style.display = 'block';
    }

    // Specific reset for delete modal
    if (modal.id === 'deleteSiswaModal') {
        const confirmCheckbox = document.getElementById('confirmDeleteSiswa');
        const confirmBtn = document.getElementById('deleteSiswaConfirmBtn');
        
        if (confirmCheckbox) confirmCheckbox.checked = false;
        if (confirmBtn) confirmBtn.disabled = true;
    }
}

function loadSiswaData(siswaId) {
    const loadingEl = document.getElementById('editLoading');
    const formContentEl = document.getElementById('editFormContent');
    
    // Show loading, hide form
    if (loadingEl) loadingEl.classList.remove('d-none');
    if (formContentEl) formContentEl.style.display = 'none';
    
    // Fetch siswa data
    fetch(`siswa.php?get_siswa=1&id=${siswaId}`)
        .then(response => {
            if (!response.ok) {
                throw new Error('Gagal memuat data siswa');
            }
            return response.json();
        })
        .then(data => {
            if (data.error) {
                throw new Error(data.error);
            }
            
            // Populate form
            document.getElementById('edit_siswa_id').value = siswaId;
            document.getElementById('edit_nis').value = data.nis || '';
            document.getElementById('edit_nama').value = data.nama || '';
            document.getElementById('edit_username').value = data.username || '';
            document.getElementById('edit_kelas_id').value = data.kelas_id || '';
            
            // Show form, hide loading
            if (loadingEl) loadingEl.classList.add('d-none');
            if (formContentEl) formContentEl.style.display = 'block';
            
            // Show modal
            const editModal = new bootstrap.Modal(document.getElementById('editSiswaModal'));
            editModal.show();
        })
        .catch(error => {
            console.error('Error:', error);
            alert(error.message);
            
            if (loadingEl) loadingEl.classList.add('d-none');
            if (formContentEl) formContentEl.style.display = 'block';
        });
}

// EMERGENCY FIX: Run this if still having issues
window.fixModalIssues = function() {
    console.log('Cleaning up modal issues...');
    
    // Remove all backdrops
    const backdrops = document.querySelectorAll('.modal-backdrop');
    backdrops.forEach(backdrop => backdrop.remove());
    
    // Reset body
    document.body.classList.remove('modal-open');
    document.body.style.overflow = '';
    document.body.style.paddingRight = '';
    
    // Hide all modals
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        modal.style.display = 'none';
        modal.classList.remove('show');
    });
    
    console.log('Cleanup complete. Backdrops removed:', backdrops.length);
};

// Auto-cleanup on page load
removeStuckBackdrops();

// Auto-close alerts
setTimeout(() => {
    document.querySelectorAll('.alert').forEach(alert => {
        const bsAlert = new bootstrap.Alert(alert);
        bsAlert.close();
    });
}, 5000);