<?php
require_once __DIR__ . '/../../app/config/config.php';
Session::checkAuth();
Session::checkRole(['admin']);

$adminModel = new AdminModel();
$tahunAktif = $adminModel->getActiveTahunAjaran();
$waliKelas = $tahunAktif ? $adminModel->getWaliKelasByTahun($tahunAktif['id']) : [];
$guru = $adminModel->getAllGuru();
$kelas = $adminModel->getAllKelas();

// Handle messages
$success = $_SESSION['success'] ?? '';
$error = $_SESSION['error'] ?? '';
unset($_SESSION['success'], $_SESSION['error']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['assign_wali'])) {
        if (!$tahunAktif) {
            $_SESSION['error'] = "Tidak ada tahun ajaran aktif!";
        } else {
            $guru_id = intval($_POST['guru_id']);
            $kelas_id = intval($_POST['kelas_id']);
            
            if (empty($guru_id) || empty($kelas_id)) {
                $_SESSION['error'] = "Guru dan kelas harus dipilih!";
            } else {
                $data = [
                    'guru_id' => $guru_id,
                    'kelas_id' => $kelas_id,
                    'tahun_ajaran_id' => $tahunAktif['id'],
                    'semester' => $tahunAktif['semester']
                ];
                
                if ($adminModel->assignWaliKelas($data)) {
                    $_SESSION['success'] = "Wali kelas berhasil ditugaskan";
                } else {
                    $_SESSION['error'] = "Gagal menugaskan wali kelas!";
                }
            }
        }
        Helper::redirect('wali_kelas.php');
    }
    
    if (isset($_POST['remove_wali'])) {
        $waliKelasId = intval($_POST['wali_kelas_id']);
        if ($adminModel->removeWaliKelas($waliKelasId)) {
            $_SESSION['success'] = "Wali kelas berhasil dihapus";
        } else {
            $_SESSION['error'] = "Gagal menghapus wali kelas!";
        }
        Helper::redirect('wali_kelas.php');
    }
}

// Get kelas with wali
$kelasWithWali = [];
foreach ($waliKelas as $wk) {
    $kelasWithWali[] = $wk['kelas_id'];
}

// Get kelas without wali
$kelasWithoutWali = array_filter($kelas, function($k) use ($kelasWithWali) {
    return !in_array($k['id'], $kelasWithWali);
});
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Wali Kelas - Sistem Penilaian</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <?php include __DIR__ . '/includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include __DIR__ . '/includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Manajemen Wali Kelas</h1>
                    <div>
                        <?php if ($tahunAktif): ?>
                        <span class="badge bg-success me-2">
                            <?= htmlspecialchars($tahunAktif['tahun_ajaran']) ?> - <?= ucfirst(htmlspecialchars($tahunAktif['semester'])) ?>
                        </span>
                        <?php endif; ?>
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#assignModal">
                            <i class="bi bi-plus-circle"></i> Tambah Wali Kelas
                        </button>
                    </div>
                </div>

                <?php if ($success): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="bi bi-check-circle-fill"></i> <?= htmlspecialchars($success) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bi bi-exclamation-triangle-fill"></i> <?= htmlspecialchars($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if (!$tahunAktif): ?>
                <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle"></i> 
                    Tidak ada tahun ajaran yang aktif. Silakan aktifkan tahun ajaran terlebih dahulu.
                </div>
                <?php endif; ?>

                <!-- Daftar Wali Kelas -->
                <div class="card shadow">
                    <div class="card-header bg-white">
                        <h5 class="card-title mb-0">
                            <i class="bi bi-people-fill"></i> Daftar Wali Kelas
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($waliKelas)): ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Kelas</th>
                                        <th>Wali Kelas</th>
                                        <th>NIP</th>
                                        <th>Tingkat</th>
                                        <th>Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($waliKelas as $wk): ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($wk['nama_kelas']) ?></strong>
                                            <?php if ($wk['jurusan']): ?>
                                            <br><small class="text-muted"><?= htmlspecialchars($wk['jurusan']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= htmlspecialchars($wk['nama_guru']) ?></td>
                                        <td><?= htmlspecialchars($wk['nip']) ?></td>
                                        <td>
                                            <span class="badge bg-secondary">Kelas <?= $wk['tingkat'] ?></span>
                                        </td>
                                        <td>
                                            <form method="POST" class="d-inline">
                                                <input type="hidden" name="wali_kelas_id" value="<?= $wk['id'] ?>">
                                                <button type="submit" name="remove_wali" 
                                                        class="btn btn-sm btn-outline-danger"
                                                        onclick="return confirm('Hapus wali kelas <?= htmlspecialchars($wk['nama_kelas']) ?>?')">
                                                    <i class="bi bi-trash"></i> Hapus
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <div class="text-center py-4">
                            <i class="bi bi-person-gear display-1 text-muted"></i>
                            <h5 class="text-muted mt-3">Belum ada penugasan wali kelas</h5>
                            <p class="text-muted">Tambahkan wali kelas untuk memulai</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Kelas Tanpa Wali -->
                <?php if (!empty($kelasWithoutWali)): ?>
                <div class="card shadow mt-4">
                    <div class="card-header bg-warning text-dark">
                        <h5 class="card-title mb-0">
                            <i class="bi bi-exclamation-triangle"></i> Kelas Belum Memiliki Wali
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <?php foreach ($kelasWithoutWali as $k): ?>
                            <div class="col-md-4 mb-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h6><?= htmlspecialchars($k['nama_kelas']) ?></h6>
                                        <p class="text-muted mb-2">Kelas <?= $k['tingkat'] ?></p>
                                        <button class="btn btn-sm btn-primary assign-btn" 
                                                data-kelas-id="<?= $k['id'] ?>"
                                                data-kelas-nama="<?= htmlspecialchars($k['nama_kelas']) ?>">
                                            <i class="bi bi-person-plus"></i> Tugaskan Wali
                                        </button>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <!-- Modal Assign Wali -->
    <div class="modal fade" id="assignModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Tugaskan Wali Kelas</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <?php if ($tahunAktif): ?>
                        <div class="mb-3">
                            <label class="form-label">Kelas</label>
                            <select name="kelas_id" class="form-select" id="modalKelasId" required>
                                <option value="">Pilih Kelas</option>
                                <?php foreach ($kelasWithoutWali as $k): ?>
                                <option value="<?= $k['id'] ?>">
                                    <?= htmlspecialchars($k['nama_kelas']) ?> (Kelas <?= $k['tingkat'] ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Guru</label>
                            <select name="guru_id" class="form-select" required>
                                <option value="">Pilih Guru</option>
                                <?php foreach ($guru as $g): ?>
                                <option value="<?= $g['id'] ?>">
                                    <?= htmlspecialchars($g['nama']) ?> (<?= htmlspecialchars($g['nip']) ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-warning">
                            Tidak ada tahun ajaran aktif
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <?php if ($tahunAktif): ?>
                        <button type="submit" name="assign_wali" class="btn btn-primary">Simpan</button>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Quick assign
        document.querySelectorAll('.assign-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const kelasId = this.getAttribute('data-kelas-id');
                const kelasNama = this.getAttribute('data-kelas-nama');
                
                document.getElementById('modalKelasId').value = kelasId;
                
                const modal = new bootstrap.Modal(document.getElementById('assignModal'));
                modal.show();
            });
        });

        // Auto close alerts
        setTimeout(() => {
            document.querySelectorAll('.alert').forEach(alert => {
                bootstrap.Alert.getOrCreateInstance(alert).close();
            });
        }, 5000);
    </script>
</body>
</html>