<?php
// DEBUG: Enable all errors
error_reporting(E_ALL);
ini_set('display_errors', 1);

try {
    require_once __DIR__ . '/../../app/config/config.php';
    Session::checkAuth();
    Session::checkRole(['admin']);

    $adminModel = new AdminModel();
    $siswa = $adminModel->getAllSiswa();
    $kelas = $adminModel->getAllKelas();

    $success = $_GET['success'] ?? '';
    $error = '';

    // Handle form submission - Add Siswa
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_siswa'])) {
        $data = [
            'nis' => Helper::sanitize($_POST['nis']),
            'nama' => Helper::sanitize($_POST['nama']),
            'username' => Helper::sanitize($_POST['username']),
            'password' => $_POST['password'],
            'kelas_id' => intval($_POST['kelas_id'])
        ];
        
        // Validation
        if (empty($data['nis']) || empty($data['nama']) || empty($data['username']) || empty($data['password'])) {
            $error = "Semua field harus diisi!";
        } else {
            $result = $adminModel->addSiswa($data);
            if ($result) {
                Helper::redirect('siswa.php?success=1');
            } else {
                $error = "Gagal menambah siswa! Username mungkin sudah digunakan.";
            }
        }
    }

    // Handle form submission - Edit Siswa
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['edit_siswa'])) {
        $siswaId = intval($_POST['siswa_id']);
        $data = [
            'nis' => Helper::sanitize($_POST['nis']),
            'nama' => Helper::sanitize($_POST['nama']),
            'username' => Helper::sanitize($_POST['username']),
            'kelas_id' => intval($_POST['kelas_id'])
        ];
        
        // Validation
        if (empty($data['nis']) || empty($data['nama']) || empty($data['username'])) {
            $error = "Semua field harus diisi!";
        } else {
            if ($adminModel->updateSiswa($siswaId, $data)) {
                Helper::redirect('siswa.php?success=2');
            } else {
                $error = "Gagal mengupdate siswa! Username mungkin sudah digunakan.";
            }
        }
    }

    // Handle delete siswa
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_siswa'])) {
        $siswaId = intval($_POST['siswa_id']);
        if ($adminModel->deleteSiswa($siswaId)) {
            Helper::redirect('siswa.php?success=3');
        } else {
            $error = "Gagal menghapus siswa! Siswa mungkin masih memiliki data terkait.";
        }
    }

    // Handle AJAX request for get siswa data
    if ($_SERVER['REQUEST_METHOD'] == 'GET' && isset($_GET['get_siswa'])) {
        $siswaId = intval($_GET['id']);
        $siswaData = $adminModel->getSiswaById($siswaId);
        
        if ($siswaData) {
            header('Content-Type: application/json');
            echo json_encode($siswaData);
            exit;
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Siswa tidak ditemukan']);
            exit;
        }
    }
} catch (Exception $e) {
    die("ERROR: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Siswa - Sistem Penilaian</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <?php include __DIR__ . '/includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include __DIR__ . '/includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Manajemen Siswa</h1>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addSiswaModal">
                        <i class="bi bi-plus-circle"></i> Tambah Siswa
                    </button>
                </div>

                <!-- DEBUG INFO -->
                <div class="alert alert-info">
                    <strong>Debug Info:</strong><br>
                    - Siswa Count: <?= count($siswa) ?><br>
                    - Kelas Count: <?= count($kelas) ?><br>
                    - Model: <?= get_class($adminModel) ?>
                </div>

                <?php if ($success == '1'): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> Siswa berhasil ditambahkan!
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php elseif ($success == '2'): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> Siswa berhasil diupdate!
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php elseif ($success == '3'): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> Siswa berhasil dihapus!
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?= $error ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <div class="card shadow">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover" id="siswaTable">
                                <thead class="table-light">
                                    <tr>
                                        <th>NIS</th>
                                        <th>Nama</th>
                                        <th>Kelas</th>
                                        <th>Username</th>
                                        <th>Status</th>
                                        <th>Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!empty($siswa)): ?>
                                        <?php foreach ($siswa as $s): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($s['nis'] ?? '') ?></td>
                                            <td><?= htmlspecialchars($s['nama'] ?? '') ?></td>
                                            <td><?= htmlspecialchars($s['nama_kelas'] ?? 'Belum ada kelas') ?></td>
                                            <td><?= htmlspecialchars($s['username'] ?? '') ?></td>
                                            <td>
                                                <span class="badge bg-<?= ($s['status'] ?? '') == 'aktif' ? 'success' : 'secondary' ?>">
                                                    <?= ucfirst($s['status'] ?? 'nonaktif') ?>
                                                </span>
                                            </td>
                                            <td>
                                                <button class="btn btn-sm btn-outline-primary edit-siswa" 
                                                        data-id="<?= $s['id'] ?? '' ?>" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#editSiswaModal">
                                                    <i class="bi bi-pencil"></i>
                                                </button>
                                                <button class="btn btn-sm btn-outline-danger delete-siswa" 
                                                        data-id="<?= $s['id'] ?? '' ?>" 
                                                        data-nama="<?= htmlspecialchars($s['nama'] ?? '') ?>">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="6" class="text-center text-muted">Tidak ada data siswa</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Add Siswa Modal -->
    <div class="modal fade" id="addSiswaModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Tambah Siswa Baru</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="nis" class="form-label">NIS</label>
                            <input type="text" class="form-control" id="nis" name="nis" required maxlength="20">
                        </div>
                        <div class="mb-3">
                            <label for="nama" class="form-label">Nama Lengkap</label>
                            <input type="text" class="form-control" id="nama" name="nama" required>
                        </div>
                        <div class="mb-3">
                            <label for="username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="username" name="username" required>
                        </div>
                        <div class="mb-3">
                            <label for="password" class="form-label">Password</label>
                            <input type="password" class="form-control" id="password" name="password" required minlength="6">
                        </div>
                        <div class="mb-3">
                            <label for="kelas_id" class="form-label">Kelas</label>
                            <select class="form-select" id="kelas_id" name="kelas_id" required>
                                <option value="">Pilih Kelas</option>
                                <?php if (!empty($kelas)): ?>
                                    <?php foreach ($kelas as $k): ?>
                                    <option value="<?= $k['id'] ?? '' ?>">
                                        <?= htmlspecialchars($k['nama_kelas'] ?? '') ?> (<?= $k['tingkat'] ?? '' ?>)
                                    </option>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="add_siswa" class="btn btn-primary">Simpan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Siswa Modal -->
    <div class="modal fade" id="editSiswaModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST" id="editSiswaForm">
                    <input type="hidden" name="siswa_id" id="edit_siswa_id">
                    <div class="modal-header">
                        <h5 class="modal-title">Edit Siswa</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div id="editLoading" class="text-center d-none">
                            <div class="spinner-border" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p>Memuat data siswa...</p>
                        </div>
                        <div id="editFormContent">
                            <div class="mb-3">
                                <label for="edit_nis" class="form-label">NIS</label>
                                <input type="text" class="form-control" id="edit_nis" name="nis" required maxlength="20">
                            </div>
                            <div class="mb-3">
                                <label for="edit_nama" class="form-label">Nama Lengkap</label>
                                <input type="text" class="form-control" id="edit_nama" name="nama" required>
                            </div>
                            <div class="mb-3">
                                <label for="edit_username" class="form-label">Username</label>
                                <input type="text" class="form-control" id="edit_username" name="username" required>
                            </div>
                            <div class="mb-3">
                                <label for="edit_kelas_id" class="form-label">Kelas</label>
                                <select class="form-select" id="edit_kelas_id" name="kelas_id" required>
                                    <option value="">Pilih Kelas</option>
                                    <?php if (!empty($kelas)): ?>
                                        <?php foreach ($kelas as $k): ?>
                                        <option value="<?= $k['id'] ?? '' ?>">
                                            <?= htmlspecialchars($k['nama_kelas'] ?? '') ?> (<?= $k['tingkat'] ?? '' ?>)
                                        </option>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="edit_siswa" class="btn btn-primary">Update Siswa</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteSiswaModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <input type="hidden" name="siswa_id" id="delete_siswa_id">
                    <div class="modal-header">
                        <h5 class="modal-title text-danger">Hapus Siswa</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-danger">
                            <i class="bi bi-exclamation-triangle"></i>
                            <strong>PERHATIAN:</strong> Tindakan ini tidak dapat dibatalkan!
                        </div>
                        <p>Anda akan menghapus siswa: <strong id="delete_siswa_nama">-</strong></p>
                        <p class="text-muted small">
                            Data yang akan dihapus:
                            <ul class="small text-muted">
                                <li>Data siswa</li>
                                <li>Data user login</li>
                                <li>Riwayat kelas</li>
                                <li>Data nilai formatif dan sumatif</li>
                            </ul>
                        </p>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="confirmDeleteSiswa">
                            <label class="form-check-label" for="confirmDeleteSiswa">
                                Saya mengerti dan ingin menghapus siswa ini
                            </label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="delete_siswa" class="btn btn-danger" id="deleteSiswaConfirmBtn" disabled>
                            <i class="bi bi-trash"></i> Hapus Siswa
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="<?= BASE_URL ?>assets/js/admin-siswa.js"></script>
</body>
</html>