<?php
require_once __DIR__ . '/../../app/config/config.php';
Session::checkAuth();
Session::checkRole(['admin']);

$adminModel = new AdminModel();
$tahunAktif = $adminModel->getActiveTahunAjaran();

// Get guru ID from URL
$guruId = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Get guru data
$guru = $adminModel->getGuruById($guruId);
if (!$guru) {
    Helper::redirect('guru.php?error=Guru tidak ditemukan');
}

// Get guru assignments
$assignments = $adminModel->getGuruAssignments($guruId);

$success = $_GET['success'] ?? '';
$error = $_GET['error'] ?? '';

// Handle remove assignment
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['remove_assignment'])) {
    $assignmentId = Helper::sanitize($_POST['assignment_id']);
    
    $result = $adminModel->removeGuruMapel($assignmentId);
    if ($result) {
        Helper::redirect("guru_detail.php?id=$guruId&success=Penugasan berhasil dihapus");
    } else {
        Helper::redirect("guru_detail.php?id=$guruId&error=Gagal menghapus penugasan");
    }
}

// Handle assign mapel
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['assign_guru'])) {
    $data = [
        'guru_id' => $guruId,
        'mapel_id' => Helper::sanitize($_POST['mapel_id']),
        'kelas_id' => Helper::sanitize($_POST['kelas_id']),
        'tahun_ajaran_id' => $tahunAktif['id']
    ];
    
    if (empty($data['mapel_id']) || empty($data['kelas_id'])) {
        $error = "Semua field harus diisi!";
    } else {
        // Cek apakah penugasan sudah ada
        $existingAssignment = $adminModel->checkExistingAssignment($data);
        
        if ($existingAssignment) {
            Helper::redirect("guru_detail.php?id=$guruId&error=Guru sudah ditugaskan untuk mapel dan kelas ini");
        } else {
            $result = $adminModel->assignGuruToMapel($data);
            if ($result) {
                Helper::redirect("guru_detail.php?id=$guruId&success=Penugasan berhasil ditambahkan");
            } else {
                Helper::redirect("guru_detail.php?id=$guruId&error=Gagal menambahkan penugasan");
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detail Guru - Sistem Penilaian</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <?php include __DIR__ . '/includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include __DIR__ . '/includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <div>
                        <h1 class="h2">Detail Guru</h1>
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb">
                                <li class="breadcrumb-item"><a href="guru.php">Manajemen Guru</a></li>
                                <li class="breadcrumb-item active"><?= htmlspecialchars($guru['nama']) ?></li>
                            </ol>
                        </nav>
                    </div>
                    <div>
                        <?php if ($tahunAktif): ?>
                        <span class="badge bg-success me-2">
                            <?= $tahunAktif['tahun_ajaran'] ?> - <?= ucfirst($tahunAktif['semester']) ?>
                        </span>
                        <?php endif; ?>
                        <a href="guru.php" class="btn btn-secondary">
                            <i class="bi bi-arrow-left"></i> Kembali
                        </a>
                    </div>
                </div>

                <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($error) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Guru Profile Card -->
                <div class="row">
                    <div class="col-md-4">
                        <div class="card shadow mb-4">
                            <div class="card-header bg-primary text-white">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-person-badge"></i> Profil Guru
                                </h5>
                            </div>
                            <div class="card-body">
                                <div class="text-center mb-3">
                                    <div class="bg-primary rounded-circle d-inline-flex align-items-center justify-content-center" style="width: 80px; height: 80px;">
                                        <i class="bi bi-person-fill text-white" style="font-size: 2rem;"></i>
                                    </div>
                                </div>
                                
                                <table class="table table-borderless">
                                    <tr>
                                        <th width="120">NIP</th>
                                        <td><?= htmlspecialchars($guru['nip']) ?></td>
                                    </tr>
                                    <tr>
                                        <th>Nama</th>
                                        <td><strong><?= htmlspecialchars($guru['nama']) ?></strong></td>
                                    </tr>
                                    <tr>
                                        <th>Username</th>
                                        <td><?= htmlspecialchars($guru['username']) ?></td>
                                    </tr>
                                    <tr>
                                        <th>Total Penugasan</th>
                                        <td>
                                            <span class="badge bg-info">
                                                <?= count($assignments) ?> mapel
                                            </span>
                                        </td>
                                    </tr>
                                </table>
                                
                                <div class="d-grid gap-2">
                                    <button class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#assignModal">
                                        <i class="bi bi-diagram-3"></i> Tambah Penugasan
                                    </button>
                                    <a href="guru.php" class="btn btn-outline-secondary">
                                        <i class="bi bi-arrow-left"></i> Kembali ke Daftar
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-8">
                        <!-- Penugasan Aktif Card -->
                        <div class="card shadow">
                            <div class="card-header bg-info text-white d-flex justify-content-between align-items-center">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-list-task"></i> Daftar Penugasan
                                </h5>
                                <span class="badge bg-light text-dark">
                                    <?= count($assignments) ?> penugasan
                                </span>
                            </div>
                            <div class="card-body">
                                <?php if (!empty($assignments)): ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover">
                                            <thead class="table-light">
                                                <tr>
                                                    <th>Tahun Ajaran</th>
                                                    <th>Mata Pelajaran</th>
                                                    <th>Kelas</th>
                                                    <th>Semester</th>
                                                    <th>Aksi</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($assignments as $assignment): ?>
                                                <tr>
                                                    <td>
                                                        <strong><?= htmlspecialchars($assignment['tahun_ajaran']) ?></strong>
                                                    </td>
                                                    <td>
                                                        <div>
                                                            <strong><?= htmlspecialchars($assignment['nama_mapel']) ?></strong>
                                                            <br>
                                                            <small class="text-muted"><?= htmlspecialchars($assignment['kode_mapel']) ?></small>
                                                        </div>
                                                    </td>
                                                    <td>
                                                        <span class="badge bg-primary">
                                                            <?= htmlspecialchars($assignment['nama_kelas']) ?> (<?= $assignment['tingkat'] ?>)
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <span class="badge bg-<?= $assignment['semester'] == 'ganjil' ? 'warning' : 'success' ?>">
                                                            <?= ucfirst($assignment['semester']) ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <form method="POST" class="d-inline" onsubmit="return confirm('Hapus penugasan ini?')">
                                                            <input type="hidden" name="assignment_id" value="<?= $assignment['id'] ?>">
                                                            <button type="submit" name="remove_assignment" class="btn btn-sm btn-outline-danger" title="Hapus Penugasan">
                                                                <i class="bi bi-trash"></i>
                                                            </button>
                                                        </form>
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="text-center py-5">
                                        <i class="bi bi-inbox display-1 text-muted"></i>
                                        <h5 class="text-muted mt-3">Belum ada penugasan</h5>
                                        <p class="text-muted">Guru ini belum ditugaskan untuk mengajar mata pelajaran apapun.</p>
                                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#assignModal">
                                            <i class="bi bi-plus-circle"></i> Tambah Penugasan Pertama
                                        </button>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Statistik Card -->
                        <div class="row mt-4">
                            <div class="col-md-6">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <i class="bi bi-book text-primary display-6"></i>
                                        <h4 class="mt-2"><?= count($assignments) ?></h4>
                                        <p class="text-muted mb-0">Total Penugasan</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <i class="bi bi-calendar-check text-success display-6"></i>
                                        <h4 class="mt-2">
                                            <?= count(array_filter($assignments, function($a) use ($tahunAktif) { 
                                                return $a['tahun_ajaran_id'] == $tahunAktif['id']; 
                                            })) ?>
                                        </h4>
                                        <p class="text-muted mb-0">Penugasan Aktif</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Assign Mapel Modal -->
    <div class="modal fade" id="assignModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Tugaskan Mata Pelajaran</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <strong>Guru:</strong> <?= htmlspecialchars($guru['nama']) ?><br>
                            <strong>Tahun Ajaran:</strong> <?= $tahunAktif ? $tahunAktif['tahun_ajaran'] : '-' ?>
                        </div>
                        <div class="mb-3">
                            <label for="mapel_id" class="form-label">Mata Pelajaran</label>
                            <select class="form-select" id="mapel_id" name="mapel_id" required>
                                <option value="">Pilih Mata Pelajaran</option>
                                <?php 
                                $mapel = $adminModel->getAllMapel();
                                foreach ($mapel as $m): 
                                ?>
                                <option value="<?= $m['id'] ?>">
                                    <?= htmlspecialchars($m['kode_mapel']) ?> - <?= htmlspecialchars($m['nama_mapel']) ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="kelas_id" class="form-label">Kelas</label>
                            <select class="form-select" id="kelas_id" name="kelas_id" required>
                                <option value="">Pilih Kelas</option>
                                <?php 
                                $kelas = $adminModel->getAllKelas();
                                foreach ($kelas as $k): 
                                ?>
                                <option value="<?= $k['id'] ?>">
                                    <?= htmlspecialchars($k['nama_kelas']) ?> (<?= $k['tingkat'] ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="assign_guru" class="btn btn-primary">Tugaskan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Tooltip initialization
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[title]'));
        const tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });

        // Reset modal form when closed
        document.getElementById('assignModal').addEventListener('hidden.bs.modal', function () {
            document.getElementById('mapel_id').value = '';
            document.getElementById('kelas_id').value = '';
        });
    </script>
</body>
</html>