<?php
require_once __DIR__ . '/../../app/config/config.php';
Session::checkAuth();
Session::checkRole(['admin']);

$adminModel = new AdminModel();
$guru = $adminModel->getAllGuru();
$tahunAktif = $adminModel->getActiveTahunAjaran();

$success = $_GET['success'] ?? '';
$error = $_GET['error'] ?? '';

// Handle add guru
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_guru'])) {
    $data = [
        'nip' => Helper::sanitize($_POST['nip']),
        'nama' => Helper::sanitize($_POST['nama']),
        'username' => Helper::sanitize($_POST['username']),
        'password' => $_POST['password']
    ];
    
    if (empty($data['nip']) || empty($data['nama']) || empty($data['username']) || empty($data['password'])) {
        $error = "Semua field harus diisi!";
    } else {
        $result = $adminModel->addGuru($data);
        if ($result) {
            Helper::redirect('guru.php?success=Guru berhasil ditambahkan');
        } else {
            $error = "Gagal menambah guru! Username mungkin sudah digunakan.";
        }
    }
}

// Handle edit guru
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['edit_guru'])) {
    $id = Helper::sanitize($_POST['guru_id']);
    $data = [
        'nip' => Helper::sanitize($_POST['nip']),
        'nama' => Helper::sanitize($_POST['nama']),
        'username' => Helper::sanitize($_POST['username'])
    ];
    
    // Jika password diisi, update password juga
    if (!empty($_POST['password'])) {
        $data['password'] = $_POST['password'];
    }
    
    if (empty($data['nip']) || empty($data['nama']) || empty($data['username'])) {
        $error = "Semua field harus diisi!";
    } else {
        $result = $adminModel->updateGuru($id, $data);
        if ($result) {
            Helper::redirect('guru.php?success=Guru berhasil diperbarui');
        } else {
            $error = "Gagal memperbarui guru! Username mungkin sudah digunakan.";
        }
    }
}

// Handle delete guru
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_guru'])) {
    $id = Helper::sanitize($_POST['guru_id']);
    
    // Cek apakah guru memiliki penugasan aktif
    $hasAssignment = $adminModel->checkGuruAssignment($id);
    
    if ($hasAssignment) {
        Helper::redirect('guru.php?error=Guru tidak dapat dihapus karena masih memiliki penugasan aktif');
    } else {
        $result = $adminModel->deleteGuru($id);
        if ($result) {
            Helper::redirect('guru.php?success=Guru berhasil dihapus');
        } else {
            Helper::redirect('guru.php?error=Gagal menghapus guru');
        }
    }
}

// Handle assign mapel
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['assign_guru'])) {
    $data = [
        'guru_id' => Helper::sanitize($_POST['guru_id']),
        'mapel_id' => Helper::sanitize($_POST['mapel_id']),
        'kelas_id' => Helper::sanitize($_POST['kelas_id']),
        'tahun_ajaran_id' => $tahunAktif['id']
    ];
    
    if (empty($data['guru_id']) || empty($data['mapel_id']) || empty($data['kelas_id'])) {
        $error = "Semua field harus diisi!";
    } else {
        // Cek apakah penugasan sudah ada
        $existingAssignment = $adminModel->checkExistingAssignment($data);
        
        if ($existingAssignment) {
            Helper::redirect('guru.php?error=Guru sudah ditugaskan untuk mapel dan kelas ini');
        } else {
            $result = $adminModel->assignGuruToMapel($data);
            if ($result) {
                Helper::redirect('guru.php?success=Penugasan berhasil ditambahkan');
            } else {
                Helper::redirect('guru.php?error=Gagal menambahkan penugasan');
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Guru - Sistem Penilaian</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <?php include __DIR__ . '/includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include __DIR__ . '/includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Manajemen Guru</h1>
                    <div>
                        <?php if ($tahunAktif): ?>
                        <span class="badge bg-success me-2">
                            <?= $tahunAktif['tahun_ajaran'] ?> - <?= ucfirst($tahunAktif['semester']) ?>
                        </span>
                        <?php endif; ?>
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addGuruModal">
                            <i class="bi bi-plus-circle"></i> Tambah Guru
                        </button>
                    </div>
                </div>

                <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($error) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <div class="card shadow">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>NIP</th>
                                        <th>Nama Lengkap</th>
                                        <th>Username</th>
                                        <th>Penugasan Aktif</th>
                                        <th>Status</th>
                                        <th>Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!empty($guru)): ?>
                                        <?php foreach ($guru as $g): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($g['nip'] ?? '') ?></td>
                                            <td>
                                                <strong><?= htmlspecialchars($g['nama'] ?? '') ?></strong>
                                            </td>
                                            <td><?= htmlspecialchars($g['username'] ?? '') ?></td>
                                            <td>
                                                <span class="badge bg-<?= $g['total_penugasan'] > 0 ? 'info' : 'secondary' ?>">
                                                    <?= $g['total_penugasan'] ?> mapel
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-success">Aktif</span>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button class="btn btn-sm btn-outline-primary edit-guru"
                                                            data-guru-id="<?= $g['id'] ?>"
                                                            data-guru-nip="<?= htmlspecialchars($g['nip']) ?>"
                                                            data-guru-nama="<?= htmlspecialchars($g['nama']) ?>"
                                                            data-guru-username="<?= htmlspecialchars($g['username']) ?>"
                                                            data-bs-toggle="modal" 
                                                            data-bs-target="#editGuruModal"
                                                            title="Edit Guru">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    <a href="guru_detail.php?id=<?= $g['id'] ?>" 
                                                       class="btn btn-sm btn-outline-info" 
                                                       title="Detail & Penugasan">
                                                        <i class="bi bi-eye"></i>
                                                    </a>
                                                    <button class="btn btn-sm btn-outline-warning assign-guru"
                                                            data-guru-id="<?= $g['id'] ?>"
                                                            data-guru-nama="<?= htmlspecialchars($g['nama']) ?>"
                                                            data-bs-toggle="modal" 
                                                            data-bs-target="#assignModal"
                                                            title="Tugaskan Mapel">
                                                        <i class="bi bi-diagram-3"></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-danger delete-guru"
                                                            data-guru-id="<?= $g['id'] ?>"
                                                            data-guru-nama="<?= htmlspecialchars($g['nama']) ?>"
                                                            data-guru-assignments="<?= $g['total_penugasan'] ?>"
                                                            data-bs-toggle="modal" 
                                                            data-bs-target="#deleteGuruModal"
                                                            title="Hapus Guru">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="6" class="text-center text-muted py-4">
                                                <i class="bi bi-person-x fs-1 d-block mb-2"></i>
                                                Tidak ada data guru
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Info Panel -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="card bg-light">
                            <div class="card-body">
                                <h6><i class="bi bi-info-circle text-primary"></i> Informasi</h6>
                                <p class="small mb-0">
                                    • Total <strong><?= count($guru) ?> guru</strong> terdaftar dalam sistem<br>
                                    • Gunakan tombol <i class="bi bi-diagram-3"></i> untuk menugaskan mapel<br>
                                    • Penugasan mengikuti tahun ajaran aktif
                                </p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card bg-light">
                            <div class="card-body">
                                <h6><i class="bi bi-shield-check text-success"></i> Status Sistem</h6>
                                <p class="small mb-0">
                                    • Tahun Ajaran: <strong><?= $tahunAktif ? $tahunAktif['tahun_ajaran'] : 'Belum diatur' ?></strong><br>
                                    • Semester: <strong><?= $tahunAktif ? ucfirst($tahunAktif['semester']) : '-' ?></strong><br>
                                    • Penugasan mengikuti tahun ajaran aktif
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Add Guru Modal -->
    <div class="modal fade" id="addGuruModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Tambah Guru Baru</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="nip" class="form-label">NIP</label>
                            <input type="text" class="form-control" id="nip" name="nip" required 
                                   placeholder="Nomor Induk Pegawai">
                        </div>
                        <div class="mb-3">
                            <label for="nama" class="form-label">Nama Lengkap</label>
                            <input type="text" class="form-control" id="nama" name="nama" required
                                   placeholder="Nama lengkap guru">
                        </div>
                        <div class="mb-3">
                            <label for="username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="username" name="username" required
                                   placeholder="Username untuk login">
                        </div>
                        <div class="mb-3">
                            <label for="password" class="form-label">Password</label>
                            <input type="password" class="form-control" id="password" name="password" required 
                                   minlength="6" placeholder="Minimal 6 karakter">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="add_guru" class="btn btn-primary">Simpan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Guru Modal -->
    <div class="modal fade" id="editGuruModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <input type="hidden" name="guru_id" id="edit_guru_id">
                    <div class="modal-header">
                        <h5 class="modal-title">Edit Data Guru</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="edit_nip" class="form-label">NIP</label>
                            <input type="text" class="form-control" id="edit_nip" name="nip" required 
                                   placeholder="Nomor Induk Pegawai">
                        </div>
                        <div class="mb-3">
                            <label for="edit_nama" class="form-label">Nama Lengkap</label>
                            <input type="text" class="form-control" id="edit_nama" name="nama" required
                                   placeholder="Nama lengkap guru">
                        </div>
                        <div class="mb-3">
                            <label for="edit_username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="edit_username" name="username" required
                                   placeholder="Username untuk login">
                        </div>
                        <div class="mb-3">
                            <label for="edit_password" class="form-label">Password Baru (Opsional)</label>
                            <input type="password" class="form-control" id="edit_password" name="password" 
                                   minlength="6" placeholder="Kosongkan jika tidak ingin mengubah">
                            <div class="form-text">Isi hanya jika ingin mengubah password</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="edit_guru" class="btn btn-primary">Update</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Guru Modal -->
    <div class="modal fade" id="deleteGuruModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <input type="hidden" name="guru_id" id="delete_guru_id">
                    <div class="modal-header">
                        <h5 class="modal-title">Konfirmasi Hapus Guru</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-warning">
                            <h6><i class="bi bi-exclamation-triangle"></i> Peringatan!</h6>
                            <p class="mb-2">Anda akan menghapus guru: <strong id="delete_guru_nama">-</strong></p>
                            <p class="mb-0" id="delete_assignment_info"></p>
                        </div>
                        <p>Data yang dihapus tidak dapat dikembalikan. Yakin ingin melanjutkan?</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="delete_guru" class="btn btn-danger">Hapus</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Assign Mapel Modal -->
    <div class="modal fade" id="assignModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <input type="hidden" name="guru_id" id="assign_guru_id">
                    <div class="modal-header">
                        <h5 class="modal-title">Tugaskan Mata Pelajaran</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            Guru: <strong id="assign_guru_nama">-</strong><br>
                            Tahun Ajaran: <strong><?= $tahunAktif ? $tahunAktif['tahun_ajaran'] : '-' ?></strong>
                        </div>
                        <div class="mb-3">
                            <label for="mapel_id" class="form-label">Mata Pelajaran</label>
                            <select class="form-select" id="mapel_id" name="mapel_id" required>
                                <option value="">Pilih Mata Pelajaran</option>
                                <?php 
                                $mapel = $adminModel->getAllMapel();
                                foreach ($mapel as $m): 
                                ?>
                                <option value="<?= $m['id'] ?>">
                                    <?= htmlspecialchars($m['kode_mapel']) ?> - <?= htmlspecialchars($m['nama_mapel']) ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="kelas_id" class="form-label">Kelas</label>
                            <select class="form-select" id="kelas_id" name="kelas_id" required>
                                <option value="">Pilih Kelas</option>
                                <?php 
                                $kelas = $adminModel->getAllKelas();
                                foreach ($kelas as $k): 
                                ?>
                                <option value="<?= $k['id'] ?>">
                                    <?= htmlspecialchars($k['nama_kelas']) ?> (<?= $k['tingkat'] ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="assign_guru" class="btn btn-primary">Tugaskan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Edit Guru functionality
        document.querySelectorAll('.edit-guru').forEach(btn => {
            btn.addEventListener('click', function() {
                const guruId = this.getAttribute('data-guru-id');
                const guruNip = this.getAttribute('data-guru-nip');
                const guruNama = this.getAttribute('data-guru-nama');
                const guruUsername = this.getAttribute('data-guru-username');
                
                document.getElementById('edit_guru_id').value = guruId;
                document.getElementById('edit_nip').value = guruNip;
                document.getElementById('edit_nama').value = guruNama;
                document.getElementById('edit_username').value = guruUsername;
                document.getElementById('edit_password').value = '';
            });
        });

        // Assign Guru functionality
        document.querySelectorAll('.assign-guru').forEach(btn => {
            btn.addEventListener('click', function() {
                const guruId = this.getAttribute('data-guru-id');
                const guruNama = this.getAttribute('data-guru-nama');
                
                document.getElementById('assign_guru_id').value = guruId;
                document.getElementById('assign_guru_nama').textContent = guruNama;
                
                // Reset form
                document.getElementById('mapel_id').value = '';
                document.getElementById('kelas_id').value = '';
            });
        });

        // Delete Guru functionality
        document.querySelectorAll('.delete-guru').forEach(btn => {
            btn.addEventListener('click', function() {
                const guruId = this.getAttribute('data-guru-id');
                const guruNama = this.getAttribute('data-guru-nama');
                const assignments = this.getAttribute('data-guru-assignments');
                
                document.getElementById('delete_guru_id').value = guruId;
                document.getElementById('delete_guru_nama').textContent = guruNama;
                
                const assignmentInfo = document.getElementById('delete_assignment_info');
                const deleteButton = document.querySelector('#deleteGuruModal button[name="delete_guru"]');
                
                if (assignments > 0) {
                    assignmentInfo.innerHTML = `<span class="text-danger"><i class="bi bi-exclamation-circle"></i> Guru ini memiliki ${assignments} penugasan aktif dan tidak dapat dihapus!</span>`;
                    deleteButton.disabled = true;
                    deleteButton.classList.add('disabled');
                } else {
                    assignmentInfo.innerHTML = '<span class="text-success"><i class="bi bi-check-circle"></i> Guru ini tidak memiliki penugasan aktif.</span>';
                    deleteButton.disabled = false;
                    deleteButton.classList.remove('disabled');
                }
            });
        });

        // Tooltip initialization
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[title]'));
        const tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });

        // Reset modals when closed
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('hidden.bs.modal', function () {
                const forms = this.querySelectorAll('form');
                forms.forEach(form => form.reset());
            });
        });
    </script>
</body>
</html>