<?php
require_once 'BaseModel.php';

class GuruModel extends BaseModel {
    
    public function __construct() {
        parent::__construct();
    }
    
    // ==================== DASHBOARD GURU ====================
    public function getDashboardStats($guruId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        $stats = [];
        
        // Total kelas yang diampu
        $sql = "SELECT COUNT(DISTINCT kelas_id) as total 
                FROM guru_mapel 
                WHERE guru_id = ? AND tahun_ajaran_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_kelas'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total mapel yang diampu
        $sql = "SELECT COUNT(DISTINCT mapel_id) as total 
                FROM guru_mapel 
                WHERE guru_id = ? AND tahun_ajaran_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_mapel'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total siswa yang diajar
        $sql = "SELECT COUNT(DISTINCT s.id) as total
                FROM siswa s
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN guru_mapel gm ON rk.kelas_id = gm.kelas_id
                WHERE gm.guru_id = ? AND gm.tahun_ajaran_id = ? AND s.status = 'aktif'";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_siswa'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total TP yang dibuat oleh guru ini
        $sql = "SELECT COUNT(DISTINCT tp.id) as total 
                FROM tujuan_pembelajaran tp
                JOIN guru_mapel gm ON (tp.mapel_id = gm.mapel_id AND tp.kelas_id = gm.kelas_id)
                WHERE gm.guru_id = ? AND tp.tahun_ajaran_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
        $stats['total_tp'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total kegiatan formatif
        $sql = "SELECT COUNT(*) as total 
                FROM kegiatan_formatif kf
                WHERE kf.guru_id = ?";
        $stmt = $this->executeQuery($sql, [$guruId]);
        $stats['total_kegiatan'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        return $stats;
    }
    
    public function getActiveTahunAjaran() {
        $sql = "SELECT * FROM tahun_ajaran WHERE status = 'aktif' LIMIT 1";
        $stmt = $this->executeQuery($sql);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }
    
    // ==================== GURU MAPEL MANAGEMENT ====================
    public function getGuruMapel($guruId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT gm.*, m.nama_mapel, m.kode_mapel, k.nama_kelas, k.tingkat, k.jurusan,
                       ta.tahun_ajaran, ta.semester,
                       (SELECT COUNT(*) FROM riwayat_kelas rk WHERE rk.kelas_id = k.id AND rk.tahun_ajaran_id = ?) as total_siswa,
                       (SELECT COUNT(*) FROM tujuan_pembelajaran tp WHERE tp.mapel_id = gm.mapel_id AND tp.kelas_id = gm.kelas_id AND tp.tahun_ajaran_id = ?) as total_tp
                FROM guru_mapel gm
                JOIN mata_pelajaran m ON gm.mapel_id = m.id
                JOIN kelas k ON gm.kelas_id = k.id
                JOIN tahun_ajaran ta ON gm.tahun_ajaran_id = ta.id
                WHERE gm.guru_id = ? AND ta.status = 'aktif'
                ORDER BY k.tingkat, k.nama_kelas, m.nama_mapel";
        
        $stmt = $this->executeQuery($sql, [$tahunAktif['id'], $tahunAktif['id'], $guruId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    // ==================== TUJUAN PEMBELAJARAN MANAGEMENT ====================
    public function getTPByMapel($mapelId, $kelasId = null) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        if ($kelasId) {
            // Filter berdasarkan mapel_id DAN kelas_id
            $sql = "SELECT tp.*, k.nama_kelas, k.tingkat, m.nama_mapel, m.kode_mapel,
                           (SELECT COUNT(*) FROM kegiatan_formatif kf WHERE kf.tp_id = tp.id) as total_kegiatan,
                           (SELECT COUNT(*) FROM nilai_formatif nf WHERE nf.tp_id = tp.id) as total_nilai_formatif,
                           (SELECT COUNT(*) FROM nilai_sumatif ns WHERE ns.tp_id = tp.id) as total_nilai_sumatif
                    FROM tujuan_pembelajaran tp
                    JOIN kelas k ON tp.kelas_id = k.id
                    JOIN mata_pelajaran m ON tp.mapel_id = m.id
                    WHERE tp.mapel_id = ? AND tp.kelas_id = ? AND tp.tahun_ajaran_id = ?
                    ORDER BY tp.created_at DESC";
            $stmt = $this->executeQuery($sql, [$mapelId, $kelasId, $tahunAktif['id']]);
        } else {
            // Fallback: ambil semua TP untuk mapel tersebut
            $sql = "SELECT tp.*, k.nama_kelas, k.tingkat, m.nama_mapel, m.kode_mapel,
                           (SELECT COUNT(*) FROM kegiatan_formatif kf WHERE kf.tp_id = tp.id) as total_kegiatan,
                           (SELECT COUNT(*) FROM nilai_formatif nf WHERE nf.tp_id = tp.id) as total_nilai_formatif,
                           (SELECT COUNT(*) FROM nilai_sumatif ns WHERE ns.tp_id = tp.id) as total_nilai_sumatif
                    FROM tujuan_pembelajaran tp
                    JOIN kelas k ON tp.kelas_id = k.id
                    JOIN mata_pelajaran m ON tp.mapel_id = m.id
                    WHERE tp.mapel_id = ? AND tp.tahun_ajaran_id = ?
                    ORDER BY tp.created_at DESC";
            $stmt = $this->executeQuery($sql, [$mapelId, $tahunAktif['id']]);
        }
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

	public function getTPById($tpId) {
		$sql = "SELECT tp.*, m.nama_mapel, m.kode_mapel, k.nama_kelas, k.tingkat, k.jurusan, k.id as kelas_id,
					   ta.tahun_ajaran, ta.semester,
					   (SELECT COUNT(*) FROM kegiatan_formatif kf WHERE kf.tp_id = tp.id) as total_kegiatan
				FROM tujuan_pembelajaran tp
				JOIN mata_pelajaran m ON tp.mapel_id = m.id
				JOIN kelas k ON tp.kelas_id = k.id
				JOIN tahun_ajaran ta ON tp.tahun_ajaran_id = ta.id
				WHERE tp.id = ?";
		
		$stmt = $this->executeQuery($sql, [$tpId]);
		return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
	}
    
    public function createTP($data) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "INSERT INTO tujuan_pembelajaran (mapel_id, kelas_id, tahun_ajaran_id, kode_tp, deskripsi, status) 
                VALUES (?, ?, ?, ?, ?, 'aktif')";
        
        $result = $this->executeQuery($sql, [
            $data['mapel_id'], 
            $data['kelas_id'],
            $tahunAktif['id'], 
            $data['kode_tp'], 
            $data['deskripsi']
        ]);
        
        return $result ? $this->db->lastInsertId() : false;
    }
    
    public function updateTP($tpId, $data) {
        $sql = "UPDATE tujuan_pembelajaran 
                SET kode_tp = ?, deskripsi = ?, status = ?
                WHERE id = ?";
        
        return $this->executeQuery($sql, [
            $data['kode_tp'],
            $data['deskripsi'],
            $data['status'],
            $tpId
        ]);
    }

    public function deleteTP($tpId) {
        // Cek apakah TP memiliki data terkait
        $checkSql = "SELECT 
                    (SELECT COUNT(*) FROM kegiatan_formatif WHERE tp_id = ?) as total_kegiatan,
                    (SELECT COUNT(*) FROM nilai_formatif WHERE tp_id = ?) as total_nilai_formatif,
                    (SELECT COUNT(*) FROM nilai_sumatif WHERE tp_id = ?) as total_nilai_sumatif";
        
        $checkStmt = $this->executeQuery($checkSql, [$tpId, $tpId, $tpId]);
        $checkData = $checkStmt ? $checkStmt->fetch(PDO::FETCH_ASSOC) : [];
        
        $totalRelated = array_sum($checkData);
        
        if ($totalRelated > 0) {
            // Jika ada data terkait, kembalikan false
            return false;
        }
        
        // Hapus TP jika tidak ada data terkait
        $sql = "DELETE FROM tujuan_pembelajaran WHERE id = ?";
        return $this->executeQuery($sql, [$tpId]);
    }

    // ==================== TP IMPORT/EXPORT & DUPLICATION ====================

    /**
     * Duplikasi TP ke kelas lain
     */
    public function duplicateTP($sourceTpId, $targetKelasId, $newKodeTP = null) {
        try {
            // Get source TP data
            $sourceTP = $this->getTPById($sourceTpId);
            if (!$sourceTP) {
                return false;
            }

            // Prepare new TP data
            $newTP = [
                'mapel_id' => $sourceTP['mapel_id'],
                'kelas_id' => $targetKelasId,
                'kode_tp' => $newKodeTP ?: $sourceTP['kode_tp'],
                'deskripsi' => $sourceTP['deskripsi'],
                'status' => 'draft'
            ];

            return $this->createTP($newTP);
        } catch (Exception $e) {
            error_log("Duplicate TP Error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Export TP data untuk mapel dan kelas tertentu
     */
    public function exportTPData($mapelId, $kelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT tp.kode_tp, tp.deskripsi, tp.status, tp.created_at,
                       m.nama_mapel, k.nama_kelas
                FROM tujuan_pembelajaran tp
                JOIN mapel m ON tp.mapel_id = m.id
                JOIN kelas k ON tp.kelas_id = k.id
                WHERE tp.mapel_id = ? AND tp.kelas_id = ? AND tp.tahun_ajaran_id = ?
                ORDER BY tp.kode_tp";
        
        $stmt = $this->executeQuery($sql, [$mapelId, $kelasId, $tahunAktif['id']]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    /**
     * Import TP data
     */
    public function importTPData($mapelId, $kelasId, $tpData) {
        try {
            $this->db->beginTransaction();
            
            $successCount = 0;
            foreach ($tpData as $tp) {
                // Skip jika kode TP sudah ada
                $checkSql = "SELECT id FROM tujuan_pembelajaran 
                            WHERE mapel_id = ? AND kelas_id = ? AND kode_tp = ? AND tahun_ajaran_id = ?";
                $tahunAktif = $this->getActiveTahunAjaran();
                $checkStmt = $this->executeQuery($checkSql, [$mapelId, $kelasId, $tp['kode_tp'], $tahunAktif['id']]);
                
                if ($checkStmt && $checkStmt->rowCount() > 0) {
                    continue; // Skip duplicate kode TP
                }
                
                $data = [
                    'mapel_id' => $mapelId,
                    'kelas_id' => $kelasId,
                    'kode_tp' => $tp['kode_tp'],
                    'deskripsi' => $tp['deskripsi'],
                    'status' => $tp['status'] ?? 'draft'
                ];
                
                if ($this->createTP($data)) {
                    $successCount++;
                }
            }
            
            $this->db->commit();
            return $successCount;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Import TP Error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get available kelas for duplication (kelas lain yang diajar guru untuk mapel yang sama)
     */
    public function getAvailableKelasForDuplication($guruId, $currentMapelId, $currentKelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT DISTINCT k.id, k.nama_kelas, k.tingkat
                FROM guru_mapel gm
                JOIN kelas k ON gm.kelas_id = k.id
                WHERE gm.guru_id = ? 
                AND gm.mapel_id = ?
                AND gm.kelas_id != ?
                AND gm.tahun_ajaran_id = ?
                ORDER BY k.tingkat, k.nama_kelas";
        
        $stmt = $this->executeQuery($sql, [$guruId, $currentMapelId, $currentKelasId, $tahunAktif['id']]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    // ==================== KEGIATAN FORMATIF MANAGEMENT ====================
    public function getKegiatanFormatifByTP($tpId) {
        $sql = "SELECT kf.*, u.nama as nama_guru,
                       (SELECT COUNT(*) FROM nilai_formatif nf WHERE nf.kegiatan_formatif_id = kf.id) as total_nilai
                FROM kegiatan_formatif kf
                JOIN guru g ON kf.guru_id = g.id
                JOIN users u ON g.user_id = u.id
                WHERE kf.tp_id = ?
                ORDER BY kf.tanggal DESC, kf.created_at DESC";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function createKegiatanFormatif($data) {
        $sql = "INSERT INTO kegiatan_formatif (tp_id, guru_id, nama_kegiatan, jenis_kegiatan, deskripsi, bobot, tanggal, status) 
                VALUES (?, ?, ?, ?, ?, ?, ?, 'aktif')";
        
        return $this->executeQuery($sql, [
            $data['tp_id'],
            $data['guru_id'],
            $data['nama_kegiatan'],
            $data['jenis_kegiatan'],
            $data['deskripsi'],
            $data['bobot'],
            $data['tanggal']
        ]);
    }

public function getKegiatanFormatifById($kegiatan_id) {
    $sql = "SELECT 
                kf.*,
                tp.kode_tp,
                tp.deskripsi as deskripsi_tp,
                tp.kelas_id,
                m.nama_mapel,
                m.kode_mapel,
                u.nama as nama_guru,  // GANTI: g.nama -> u.nama
                kl.nama_kelas,
                kl.tingkat
            FROM kegiatan_formatif kf
            JOIN tujuan_pembelajaran tp ON kf.tp_id = tp.id
            JOIN mata_pelajaran m ON tp.mapel_id = m.id
            JOIN guru g ON kf.guru_id = g.id
            JOIN users u ON g.user_id = u.id  // TAMBAH JOIN INI
            JOIN kelas kl ON tp.kelas_id = kl.id
            WHERE kf.id = ?";
    
    $stmt = $this->executeQuery($sql, [$kegiatan_id]);
    return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : null;
}

public function getKegiatanFormatifWithKelas($kegiatan_id) {
    $sql = "SELECT 
                kf.*,
                tp.kode_tp,
                tp.deskripsi as deskripsi_tp,
                tp.kelas_id,
                m.nama_mapel,
                m.kode_mapel,
                kl.nama_kelas,
                kl.tingkat
            FROM kegiatan_formatif kf
            JOIN tujuan_pembelajaran tp ON kf.tp_id = tp.id
            JOIN mata_pelajaran m ON tp.mapel_id = m.id
            JOIN kelas kl ON tp.kelas_id = kl.id
            WHERE kf.id = ?";
    
    $stmt = $this->executeQuery($sql, [$kegiatan_id]);
    return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : null;
}
    public function updateKegiatanFormatif($kegiatanId, $data) {
        $sql = "UPDATE kegiatan_formatif SET 
                nama_kegiatan = ?, jenis_kegiatan = ?, deskripsi = ?, 
                bobot = ?, tanggal = ?, status = ?, updated_at = NOW()
                WHERE id = ?";
        
        return $this->executeQuery($sql, [
            $data['nama_kegiatan'],
            $data['jenis_kegiatan'],
            $data['deskripsi'],
            $data['bobot'],
            $data['tanggal'],
            $data['status'],
            $kegiatanId
        ]);
    }

    public function deleteKegiatanFormatif($kegiatanId) {
        // Cek apakah kegiatan memiliki nilai
        $checkSql = "SELECT COUNT(*) as total_nilai FROM nilai_formatif WHERE kegiatan_formatif_id = ?";
        $checkStmt = $this->executeQuery($checkSql, [$kegiatanId]);
        $checkData = $checkStmt ? $checkStmt->fetch(PDO::FETCH_ASSOC) : [];
        
        if ($checkData['total_nilai'] > 0) {
            return false;
        }
        
        $sql = "DELETE FROM kegiatan_formatif WHERE id = ?";
        return $this->executeQuery($sql, [$kegiatanId]);
    }
	

    // ==================== NILAI FORMATIF TERHUBUNG KEGIATAN ====================
	// ==================== NILAI FORMATIF TERHUBUNG KEGIATAN ====================
	public function inputNilaiFormatifWithKegiatan($data) {
		try {
			$checkSql = "SELECT id FROM nilai_formatif 
						 WHERE kegiatan_formatif_id = ? AND siswa_id = ?";
			$checkStmt = $this->executeQuery($checkSql, [
				$data['kegiatan_formatif_id'], $data['siswa_id']
			]);
			
			if ($checkStmt && $checkStmt->rowCount() > 0) {
				// UPDATE existing record
				$sql = "UPDATE nilai_formatif 
						SET nilai = ?, deskripsi = ?, updated_by = ?, updated_at = NOW(),
							jenis = ?, tp_id = ?
						WHERE kegiatan_formatif_id = ? AND siswa_id = ?";
				$params = [
					$data['nilai'], 
					$data['deskripsi'], 
					$data['updated_by'],
					$data['jenis'],
					$data['tp_id'],
					$data['kegiatan_formatif_id'], 
					$data['siswa_id']
				];
			} else {
				// INSERT new record
				$sql = "INSERT INTO nilai_formatif (kegiatan_formatif_id, tp_id, siswa_id, jenis, nilai, deskripsi, updated_by) 
						VALUES (?, ?, ?, ?, ?, ?, ?)";
				$params = [
					$data['kegiatan_formatif_id'], 
					$data['tp_id'], 
					$data['siswa_id'],
					$data['jenis'], 
					$data['nilai'], 
					$data['deskripsi'], 
					$data['updated_by']
				];
			}
			
			$result = $this->executeQuery($sql, $params);
			return $result !== false;
			
		} catch (Exception $e) {
			error_log("Error inputNilaiFormatifWithKegiatan: " . $e->getMessage());
			return false;
		}
	}

    public function getNilaiFormatifByKegiatan($kegiatanId) {
        $sql = "SELECT nf.*, s.nis, u.nama as nama_siswa, k.nama_kelas
                FROM nilai_formatif nf
                JOIN siswa s ON nf.siswa_id = s.id
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN kelas k ON rk.kelas_id = k.id
                JOIN tahun_ajaran ta ON rk.tahun_ajaran_id = ta.id
                WHERE nf.kegiatan_formatif_id = ? AND ta.status = 'aktif'
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql, [$kegiatanId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function getNilaiFormatifBySiswaTP($siswaId, $tpId) {
        $sql = "SELECT nf.*, kf.nama_kegiatan, kf.jenis_kegiatan, kf.bobot, kf.tanggal
                FROM nilai_formatif nf
                JOIN kegiatan_formatif kf ON nf.kegiatan_formatif_id = kf.id
                WHERE nf.siswa_id = ? AND kf.tp_id = ?
                ORDER BY kf.tanggal";
        
        $stmt = $this->executeQuery($sql, [$siswaId, $tpId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    // ==================== NILAI MANAGEMENT (LEGACY) ====================
    public function inputNilaiFormatif($data) {
        $checkSql = "SELECT id FROM nilai_formatif 
                     WHERE tp_id = ? AND siswa_id = ? AND jenis = ?";
        $checkStmt = $this->executeQuery($checkSql, [
            $data['tp_id'], $data['siswa_id'], $data['jenis']
        ]);
        
        if ($checkStmt && $checkStmt->rowCount() > 0) {
            $sql = "UPDATE nilai_formatif 
                    SET nilai = ?, deskripsi = ?, updated_by = ?, updated_at = NOW() 
                    WHERE tp_id = ? AND siswa_id = ? AND jenis = ?";
        } else {
            $sql = "INSERT INTO nilai_formatif (tp_id, siswa_id, jenis, nilai, deskripsi, updated_by) 
                    VALUES (?, ?, ?, ?, ?, ?)";
        }
        
        return $this->executeQuery($sql, [
            $data['nilai'], $data['deskripsi'], $data['updated_by'],
            $data['tp_id'], $data['siswa_id'], $data['jenis']
        ]);
    }
    
    public function getNilaiSumatifTP($tpId, $siswaId = null) {
        $sql = "SELECT ns.*, u.nama as nama_siswa, s.nis, k.nama_kelas
                FROM nilai_sumatif ns
                JOIN siswa s ON ns.siswa_id = s.id
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN kelas k ON rk.kelas_id = k.id
                JOIN tahun_ajaran ta ON rk.tahun_ajaran_id = ta.id
                WHERE ns.tp_id = ? AND ta.status = 'aktif'" . 
                ($siswaId ? " AND ns.siswa_id = ?" : "") . "
                ORDER BY u.nama";
        
        $params = [$tpId];
        if ($siswaId) {
            $params[] = $siswaId;
        }
        
        $stmt = $this->executeQuery($sql, $params);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
	// ==================== NILAI SUMATIF MANAGEMENT ====================
	public function inputNilaiSumatifTP($data) {
		try {
			$checkSql = "SELECT id FROM nilai_sumatif 
						 WHERE tp_id = ? AND siswa_id = ?";
			$checkStmt = $this->executeQuery($checkSql, [
				$data['tp_id'], $data['siswa_id']
			]);
			
			if ($checkStmt && $checkStmt->rowCount() > 0) {
				// UPDATE existing record
				$sql = "UPDATE nilai_sumatif 
						SET nilai = ?, jenis_penilaian = ?, deskripsi = ?, 
							updated_by = ?, updated_at = NOW() 
						WHERE tp_id = ? AND siswa_id = ?";
				$params = [
					$data['nilai'],
					$data['jenis_penilaian'],
					$data['deskripsi'],
					$data['updated_by'],
					$data['tp_id'],
					$data['siswa_id']
				];
			} else {
				// INSERT new record
				$sql = "INSERT INTO nilai_sumatif (tp_id, siswa_id, nilai, jenis_penilaian, deskripsi, updated_by) 
						VALUES (?, ?, ?, ?, ?, ?)";
				$params = [
					$data['tp_id'],
					$data['siswa_id'],
					$data['nilai'],
					$data['jenis_penilaian'],
					$data['deskripsi'],
					$data['updated_by']
				];
			}
			
			$result = $this->executeQuery($sql, $params);
			return $result !== false;
			
		} catch (Exception $e) {
			error_log("Error inputNilaiSumatifTP: " . $e->getMessage());
			return false;
		}
	}
    
    
/**
 * Input/Update nilai sumatif akhir (SAS)
 */
public function inputNilaiSumatifAkhir($data) {
    try {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $checkSql = "SELECT id FROM nilai_sumatif_akhir 
                     WHERE siswa_id = ? AND mapel_id = ? AND tahun_ajaran_id = ?";
        $checkStmt = $this->executeQuery($checkSql, [
            $data['siswa_id'], $data['mapel_id'], $tahunAktif['id']
        ]);
        
        if ($checkStmt && $checkStmt->rowCount() > 0) {
            // UPDATE existing record
            $sql = "UPDATE nilai_sumatif_akhir 
                    SET nilai = ?, updated_by = ?, updated_at = NOW() 
                    WHERE siswa_id = ? AND mapel_id = ? AND tahun_ajaran_id = ?";
            $params = [
                $data['nilai'],
                $data['updated_by'],
                $data['siswa_id'],
                $data['mapel_id'],
                $tahunAktif['id']
            ];
        } else {
            // INSERT new record
            $sql = "INSERT INTO nilai_sumatif_akhir (siswa_id, mapel_id, tahun_ajaran_id, semester, nilai, updated_by) 
                    VALUES (?, ?, ?, ?, ?, ?)";
            $params = [
                $data['siswa_id'],
                $data['mapel_id'],
                $tahunAktif['id'],
                $tahunAktif['semester'],
                $data['nilai'],
                $data['updated_by']
            ];
        }
        
        $result = $this->executeQuery($sql, $params);
        return $result !== false;
        
    } catch (Exception $e) {
        error_log("Error inputNilaiSumatifAkhir: " . $e->getMessage());
        return false;
    }
}

    // ==================== REKAP NILAI TERPADU ====================
    public function getRekapNilaiTerpadu($kelasId, $mapelId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT 
                    s.id as siswa_id, u.nama as nama_siswa, s.nis,
                    tp.id as tp_id, tp.kode_tp, tp.deskripsi as deskripsi_tp,
                    kf.nama_kegiatan, kf.jenis_kegiatan, kf.bobot as bobot_kegiatan,
                    nf.nilai as nilai_formatif, nf.deskripsi as deskripsi_formatif,
                    ns.nilai as nilai_sumatif_tp, ns.deskripsi as deskripsi_sumatif,
                    nsa.nilai as nilai_sumatif_akhir
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN tujuan_pembelajaran tp ON tp.mapel_id = ? AND tp.tahun_ajaran_id = ? AND tp.kelas_id = ?
                LEFT JOIN kegiatan_formatif kf ON kf.tp_id = tp.id
                LEFT JOIN nilai_formatif nf ON (nf.kegiatan_formatif_id = kf.id AND nf.siswa_id = s.id)
                LEFT JOIN nilai_sumatif ns ON (tp.id = ns.tp_id AND s.id = ns.siswa_id)
                LEFT JOIN nilai_sumatif_akhir nsa ON (s.id = nsa.siswa_id AND nsa.mapel_id = ? AND nsa.tahun_ajaran_id = ?)
                WHERE rk.kelas_id = ? AND rk.tahun_ajaran_id = ? AND s.status = 'aktif'
                ORDER BY u.nama, tp.kode_tp, kf.nama_kegiatan";
        
        $stmt = $this->executeQuery($sql, [
            $mapelId, $tahunAktif['id'], $kelasId,
            $mapelId, $tahunAktif['id'],
            $kelasId, $tahunAktif['id']
        ]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function getRekapNilaiKelas($kelasId, $mapelId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT s.id as siswa_id, u.nama as nama_siswa, s.nis,
                       tp.kode_tp, tp.deskripsi as deskripsi_tp,
                       nf.jenis, nf.nilai as nilai_formatif, nf.deskripsi as deskripsi_formatif,
                       ns.nilai as nilai_sumatif_tp, ns.deskripsi as deskripsi_sumatif,
                       nsa.nilai as nilai_sumatif_akhir
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN tujuan_pembelajaran tp ON tp.mapel_id = ? AND tp.tahun_ajaran_id = ? AND tp.kelas_id = ?
                LEFT JOIN nilai_formatif nf ON (tp.id = nf.tp_id AND s.id = nf.siswa_id)
                LEFT JOIN nilai_sumatif ns ON (tp.id = ns.tp_id AND s.id = ns.siswa_id)
                LEFT JOIN nilai_sumatif_akhir nsa ON (s.id = nsa.siswa_id AND nsa.mapel_id = ? AND nsa.tahun_ajaran_id = ?)
                WHERE rk.kelas_id = ? AND rk.tahun_ajaran_id = ? AND s.status = 'aktif'
                ORDER BY u.nama, tp.kode_tp, nf.jenis";
        
        $stmt = $this->executeQuery($sql, [
            $mapelId, $tahunAktif['id'], $kelasId,
            $mapelId, $tahunAktif['id'],
            $kelasId, $tahunAktif['id']
        ]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
	
	/**
 * Get mapel by guru (disesuaikan dengan struktur database)
 */
public function getMapelByGuru($guruId) {
    $tahunAktif = $this->getActiveTahunAjaran();
    
    $sql = "SELECT DISTINCT m.id, m.nama_mapel, m.kode_mapel 
            FROM mata_pelajaran m 
            INNER JOIN guru_mapel gm ON m.id = gm.mapel_id 
            WHERE gm.guru_id = ? AND gm.tahun_ajaran_id = ?
            ORDER BY m.nama_mapel";
    
    $stmt = $this->executeQuery($sql, [$guruId, $tahunAktif['id']]);
    return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
}

/**
 * Get kelas by mapel dan guru
 */
public function getKelasByMapelGuru($guruId, $mapelId) {
    $tahunAktif = $this->getActiveTahunAjaran();
    
    $sql = "SELECT DISTINCT k.id, k.nama_kelas, k.tingkat 
            FROM kelas k 
            INNER JOIN guru_mapel gm ON k.id = gm.kelas_id 
            WHERE gm.guru_id = ? AND gm.mapel_id = ? AND gm.tahun_ajaran_id = ?
            ORDER BY k.tingkat, k.nama_kelas";
    
    $stmt = $this->executeQuery($sql, [$guruId, $mapelId, $tahunAktif['id']]);
    return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
}

    // ==================== SISWA MANAGEMENT ====================
    public function getSiswaByKelas($kelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT s.*, u.nama, u.username, k.nama_kelas, k.tingkat
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN riwayat_kelas rk ON s.id = rk.siswa_id
                JOIN kelas k ON rk.kelas_id = k.id
                WHERE rk.kelas_id = ? AND rk.tahun_ajaran_id = ? AND s.status = 'aktif'
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql, [$kelasId, $tahunAktif['id']]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
	
	public function getSiswaByKelasForNilai($kelasId) {
		$tahunAktif = $this->getActiveTahunAjaran();
		
		$sql = "SELECT DISTINCT 
					s.id,
					s.nis,
					u.nama,
					u.id as user_id,
					rk.kelas_id,
					k.nama_kelas,
					k.tingkat,
					ta.tahun_ajaran,
					ta.semester
				FROM siswa s
				JOIN users u ON s.user_id = u.id
				JOIN riwayat_kelas rk ON s.id = rk.siswa_id
				JOIN kelas k ON rk.kelas_id = k.id
				JOIN tahun_ajaran ta ON rk.tahun_ajaran_id = ta.id
				WHERE rk.kelas_id = ? 
				AND ta.status = 'aktif'
				AND s.status = 'aktif'
				ORDER BY u.nama";
		
		$stmt = $this->executeQuery($sql, [$kelasId]);
		return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
	}

    // ==================== WALI KELAS MANAGEMENT ====================
    public function getWaliKelasByGuru($guruId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT wk.*, k.nama_kelas, k.tingkat, k.jurusan, ta.tahun_ajaran, ta.semester,
                       (SELECT COUNT(*) FROM riwayat_kelas rk WHERE rk.kelas_id = k.id AND rk.tahun_ajaran_id = ?) as total_siswa
                FROM wali_kelas wk
                JOIN kelas k ON wk.kelas_id = k.id
                JOIN tahun_ajaran ta ON wk.tahun_ajaran_id = ta.id
                WHERE wk.guru_id = ? AND ta.status = 'aktif'";
        
        $stmt = $this->executeQuery($sql, [$tahunAktif['id'], $guruId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }

    // ==================== VALIDASI AKSES GURU ====================
    public function validateGuruMapelAccess($guruId, $mapelId, $kelasId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT id FROM guru_mapel 
                WHERE guru_id = ? AND mapel_id = ? AND kelas_id = ? AND tahun_ajaran_id = ?";
        
        $stmt = $this->executeQuery($sql, [
            $guruId, $mapelId, $kelasId, $tahunAktif['id']
        ]);
        
        return $stmt && $stmt->rowCount() > 0;
    }

    public function validateGuruTPAccess($guruId, $tpId) {
		$tahunAktif = $this->getActiveTahunAjaran();
		
		$sql = "SELECT gm.id 
				FROM guru_mapel gm
				JOIN tujuan_pembelajaran tp ON (gm.mapel_id = tp.mapel_id AND gm.kelas_id = tp.kelas_id AND gm.tahun_ajaran_id = tp.tahun_ajaran_id)
				WHERE gm.guru_id = ? AND tp.id = ? AND gm.tahun_ajaran_id = ?";
		
		$stmt = $this->executeQuery($sql, [$guruId, $tpId, $tahunAktif['id']]);
		return $stmt && $stmt->rowCount() > 0;
	}

    public function validateGuruKegiatanAccess($guruId, $kegiatanId) {
        $sql = "SELECT kf.id 
                FROM kegiatan_formatif kf
                WHERE kf.id = ? AND kf.guru_id = ?";
        
        $stmt = $this->executeQuery($sql, [$kegiatanId, $guruId]);
        return $stmt && $stmt->rowCount() > 0;
    }

    // ==================== UTILITY METHODS ====================
    public function getRataRataNilaiFormatifByTP($tpId) {
        $sql = "SELECT AVG(nf.nilai) as rata_rata, COUNT(nf.id) as total_nilai
                FROM nilai_formatif nf
                WHERE nf.tp_id = ?";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : ['rata_rata' => 0, 'total_nilai' => 0];
    }

    public function getRataRataNilaiSumatifByTP($tpId) {
        $sql = "SELECT AVG(ns.nilai) as rata_rata, COUNT(ns.id) as total_nilai
                FROM nilai_sumatif ns
                WHERE ns.tp_id = ?";
        
        $stmt = $this->executeQuery($sql, [$tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : ['rata_rata' => 0, 'total_nilai' => 0];
    }


public function getMapelById($mapelId) {
    $sql = "SELECT * FROM mata_pelajaran WHERE id = ?";
    $stmt = $this->executeQuery($sql, [$mapelId]);
    return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
}

/**
 * Get kelas by ID
 */
public function getKelasById($kelasId) {
    $sql = "SELECT * FROM kelas WHERE id = ?";
    $stmt = $this->executeQuery($sql, [$kelasId]);
    return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
}

/**
 * Get tahun ajaran by ID
 */
public function getTahunAjaranById($tahunAjaranId) {
    $sql = "SELECT * FROM tahun_ajaran WHERE id = ?";
    $stmt = $this->executeQuery($sql, [$tahunAjaranId]);
    return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
}

/**
 * Get nilai sumatif akhir dengan data siswa
 */
public function getNilaiSumatifAkhir($mapelId, $kelasId) {
    $tahunAktif = $this->getActiveTahunAjaran();
    
    $sql = "SELECT nsa.*, s.id as siswa_id, u.nama as nama_siswa, s.nis
            FROM nilai_sumatif_akhir nsa
            JOIN siswa s ON nsa.siswa_id = s.id
            JOIN users u ON s.user_id = u.id
            JOIN riwayat_kelas rk ON s.id = rk.siswa_id
            WHERE nsa.mapel_id = ? AND nsa.tahun_ajaran_id = ? 
            AND rk.kelas_id = ? AND rk.tahun_ajaran_id = ?
            ORDER BY u.nama";
    
    $stmt = $this->executeQuery($sql, [
        $mapelId, $tahunAktif['id'], $kelasId, $tahunAktif['id']
    ]);
    return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
}
    public function getProgressTP($tpId) {
        $tahunAktif = $this->getActiveTahunAjaran();
        
        $sql = "SELECT 
                    (SELECT COUNT(DISTINCT s.id) 
                     FROM siswa s 
                     JOIN riwayat_kelas rk ON s.id = rk.siswa_id 
                     JOIN tujuan_pembelajaran tp ON rk.kelas_id = tp.kelas_id 
                     WHERE tp.id = ? AND rk.tahun_ajaran_id = ?) as total_siswa,
                    (SELECT COUNT(DISTINCT nf.siswa_id) 
                     FROM nilai_formatif nf 
                     WHERE nf.tp_id = ?) as siswa_terinput_formatif,
                    (SELECT COUNT(DISTINCT ns.siswa_id) 
                     FROM nilai_sumatif ns 
                     WHERE ns.tp_id = ?) as siswa_terinput_sumatif";
        
        $stmt = $this->executeQuery($sql, [$tpId, $tahunAktif['id'], $tpId, $tpId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : [
            'total_siswa' => 0,
            'siswa_terinput_formatif' => 0,
            'siswa_terinput_sumatif' => 0
        ];
    }
}
?>