<?php
class AdminModel extends BaseModel {
    
    public function __construct() {
        parent::__construct();
    }
    
    // ==================== DASHBOARD & STATISTICS ====================
    public function getSystemStats() {
        $tahunAktif = $this->getActiveTahunAjaran();
        $stats = [];
        
        // Total Siswa Aktif
        $sql = "SELECT COUNT(*) as total FROM siswa WHERE status = 'aktif'";
        $stmt = $this->executeQuery($sql);
        $stats['total_siswa'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total Guru
        $sql = "SELECT COUNT(*) as total FROM guru";
        $stmt = $this->executeQuery($sql);
        $stats['total_guru'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total Kelas
        $sql = "SELECT COUNT(*) as total FROM kelas";
        $stmt = $this->executeQuery($sql);
        $stats['total_kelas'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Total Mata Pelajaran
        $sql = "SELECT COUNT(*) as total FROM mata_pelajaran";
        $stmt = $this->executeQuery($sql);
        $stats['total_mapel'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        
        // Guru dengan penugasan di tahun aktif
        if ($tahunAktif) {
            $sql = "SELECT COUNT(DISTINCT guru_id) as total 
                    FROM guru_mapel 
                    WHERE tahun_ajaran_id = ?";
            $stmt = $this->executeQuery($sql, [$tahunAktif['id']]);
            $stats['guru_aktif'] = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC)['total'] : 0;
        } else {
            $stats['guru_aktif'] = 0;
        }
        
        return $stats;
    }
    
    public function getRecentActivity($limit = 10) {
        $sql = "SELECT la.*, u.nama, u.username 
                FROM log_aktivitas la
                JOIN users u ON la.user_id = u.id
                ORDER BY la.created_at DESC 
                LIMIT ?";
        
        $stmt = $this->executeQuery($sql, [$limit]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    // ==================== TAHUN AJARAN MANAGEMENT ====================
    public function getAllTahunAjaran() {
        $sql = "SELECT * FROM tahun_ajaran ORDER BY tahun_ajaran DESC, semester DESC";
        $stmt = $this->executeQuery($sql);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    public function getActiveTahunAjaran() {
        $sql = "SELECT * FROM tahun_ajaran WHERE status = 'aktif' LIMIT 1";
        $stmt = $this->executeQuery($sql);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }
    
    public function activateTahunAjaran($tahunId) {
        try {
            // Start transaction
            $this->db->beginTransaction();
            
            // Nonaktifkan semua tahun ajaran
            $this->executeQuery("UPDATE tahun_ajaran SET status = 'nonaktif'");
            
            // Aktifkan tahun ajaran yang dipilih
            $result = $this->executeQuery(
                "UPDATE tahun_ajaran SET status = 'aktif' WHERE id = ?",
                [$tahunId]
            );
            
            // Auto kenaikan kelas untuk siswa
            $this->autoKenaikanKelas($tahunId);
            
            $this->db->commit();
            return $result !== false;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Activate Tahun Ajaran Error: " . $e->getMessage());
            return false;
        }
    }
    
   private function autoKenaikanKelas($tahunAjaranId) {
    $tahunBaru = $this->executeQuery(
        "SELECT * FROM tahun_ajaran WHERE id = ?", 
        [$tahunAjaranId]
    )->fetch(PDO::FETCH_ASSOC);
    
    // Dapatkan tahun ajaran sebelumnya yang aktif
    $tahunSebelumnya = $this->executeQuery(
        "SELECT * FROM tahun_ajaran WHERE status = 'nonaktif' ORDER BY id DESC LIMIT 1"
    )->fetch(PDO::FETCH_ASSOC);
    
    // Hanya naikkan kelas jika:
    // 1. Semester ganjil DAN 
    // 2. Tahun ajaran berbeda dengan sebelumnya (tahun baru)
    if ($tahunBaru['semester'] == 'ganjil' && $tahunSebelumnya) {
        $tahunBaruParts = explode('/', $tahunBaru['tahun_ajaran']);
        $tahunSebelumnyaParts = explode('/', $tahunSebelumnya['tahun_ajaran']);
        
        // Cek apakah tahun ajaran benar-benar baru (2024/2025 → 2025/2026)
        if ($tahunBaruParts[0] != $tahunSebelumnyaParts[0]) {
            $sql = "UPDATE kelas SET tingkat = tingkat + 1 WHERE tingkat < 9";
            $this->executeQuery($sql);
            
            // Update riwayat kelas untuk siswa
            $sql = "INSERT INTO riwayat_kelas (siswa_id, kelas_id, tahun_ajaran_id, semester)
                    SELECT s.id, 
                           CASE 
                               WHEN k.tingkat = 9 THEN k.id  // Tetap di kelas 9
                               ELSE (SELECT id FROM kelas WHERE tingkat = k.tingkat + 1 AND jurusan = k.jurusan LIMIT 1)
                           END as new_kelas_id,
                           ?, 'ganjil'
                    FROM siswa s
                    JOIN kelas k ON s.kelas_id = k.id
                    WHERE s.status = 'aktif'";
            $this->executeQuery($sql, [$tahunAjaranId]);
            
            error_log("Kenaikan kelas dilakukan: " . $tahunSebelumnya['tahun_ajaran'] . " → " . $tahunBaru['tahun_ajaran']);
        } else {
            error_log("Tidak ada kenaikan kelas: Tahun ajaran sama (" . $tahunBaru['tahun_ajaran'] . ")");
        }
    } else {
        error_log("Tidak ada kenaikan kelas: Semester " . $tahunBaru['semester']);
    }
	}
    
    public function addTahunAjaran($data) {
        $sql = "INSERT INTO tahun_ajaran (tahun_ajaran, semester) 
                VALUES (?, ?)";
        
        return $this->executeQuery($sql, [
            $data['tahun_ajaran'],
            $data['semester']
        ]);
    }
    
    // ==================== SISWA MANAGEMENT ====================
        // ==================== SISWA MANAGEMENT ====================
    public function getAllSiswa() {
        $tahunAktif = $this->getActiveTahunAjaran();
        $tahunId = $tahunAktif ? $tahunAktif['id'] : 0;
        
        $sql = "SELECT s.*, u.nama, u.username, k.nama_kelas, k.tingkat, k.jurusan,
                       rk.tahun_ajaran_id, rk.semester as current_semester
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN kelas k ON s.kelas_id = k.id
                LEFT JOIN riwayat_kelas rk ON (s.id = rk.siswa_id AND rk.tahun_ajaran_id = ?)
                WHERE s.status = 'aktif'
                ORDER BY k.tingkat, k.nama_kelas, u.nama";
        
        $stmt = $this->executeQuery($sql, [$tahunId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function getSiswaById($siswaId) {
        $sql = "SELECT s.*, u.nama, u.username, u.id as user_id, k.nama_kelas, k.tingkat, k.jurusan, k.id as kelas_id
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                JOIN kelas k ON s.kelas_id = k.id
                WHERE s.id = ?";
        
        $stmt = $this->executeQuery($sql, [$siswaId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }

    public function updateSiswa($siswaId, $data) {
        try {
            // Get current student data
            $currentData = $this->getSiswaById($siswaId);
            if (!$currentData) {
                return false;
            }
            
            // Update user data
            $userSql = "UPDATE users SET nama = ?, username = ? WHERE id = ?";
            $userStmt = $this->executeQuery($userSql, [
                $data['nama'],
                $data['username'],
                $currentData['user_id']
            ]);
            
            if (!$userStmt) return false;
            
            // Update siswa data
            $siswaSql = "UPDATE siswa SET nis = ?, kelas_id = ? WHERE id = ?";
            $siswaStmt = $this->executeQuery($siswaSql, [
                $data['nis'],
                $data['kelas_id'],
                $siswaId
            ]);
            
            return $siswaStmt !== false;
            
        } catch (Exception $e) {
            error_log("Update Siswa Error: " . $e->getMessage());
            return false;
        }
    }

    public function deleteSiswa($siswaId) {
        try {
            // Nonaktifkan siswa (soft delete)
            $sql = "UPDATE siswa SET status = 'nonaktif' WHERE id = ?";
            $result = $this->executeQuery($sql, [$siswaId]);
            
            return $result !== false;
            
        } catch (Exception $e) {
            error_log("Delete Siswa Error: " . $e->getMessage());
            return false;
        }
    }
    
    public function addSiswa($data) {
        try {
            $tahunAktif = $this->getActiveTahunAjaran();
            if (!$tahunAktif) {
                throw new Exception("Tidak ada tahun ajaran aktif");
            }
            
            // Step 1: Create user
            $userSql = "INSERT INTO users (username, password, level, nama) 
                        VALUES (?, ?, 'siswa', ?)";
            $userStmt = $this->executeQuery($userSql, [
                $data['username'], 
                password_hash($data['password'], PASSWORD_DEFAULT), 
                $data['nama']
            ]);
            
            if (!$userStmt) throw new Exception("Failed to create user");
            
            $userId = $this->db->lastInsertId();
            
            // Step 2: Create siswa record
            $siswaSql = "INSERT INTO siswa (user_id, nis, kelas_id, status) 
                         VALUES (?, ?, ?, 'aktif')";
            $siswaStmt = $this->executeQuery($siswaSql, [$userId, $data['nis'], $data['kelas_id']]);
            
            if (!$siswaStmt) throw new Exception("Failed to create siswa record");
            
            $siswaId = $this->db->lastInsertId();
            
            // Step 3: Add to riwayat_kelas
            $riwayatSql = "INSERT INTO riwayat_kelas (siswa_id, kelas_id, tahun_ajaran_id, semester) 
                           VALUES (?, ?, ?, ?)";
            $this->executeQuery($riwayatSql, [
                $siswaId, 
                $data['kelas_id'], 
                $tahunAktif['id'],
                $tahunAktif['semester']
            ]);
            
            return $siswaId;
            
        } catch (Exception $e) {
            error_log("Add Siswa Error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getRiwayatKelas($siswaId) {
        $sql = "SELECT rk.*, ta.tahun_ajaran, ta.semester, k.nama_kelas, k.tingkat, k.jurusan
                FROM riwayat_kelas rk
                JOIN tahun_ajaran ta ON rk.tahun_ajaran_id = ta.id
                JOIN kelas k ON rk.kelas_id = k.id
                WHERE rk.siswa_id = ?
                ORDER BY ta.tahun_ajaran DESC, ta.semester DESC";
        
        $stmt = $this->executeQuery($sql, [$siswaId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    // ==================== GURU MANAGEMENT ====================
    public function getAllGuru() {
        $sql = "SELECT g.*, u.nama, u.username,
                       (SELECT COUNT(*) FROM guru_mapel gm WHERE gm.guru_id = g.id) as total_penugasan
                FROM guru g
                JOIN users u ON g.user_id = u.id
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    public function addGuru($data) {
        try {
            // Step 1: Create user
            $userSql = "INSERT INTO users (username, password, level, nama) 
                        VALUES (?, ?, 'guru', ?)";
            $userStmt = $this->executeQuery($userSql, [
                $data['username'], 
                password_hash($data['password'], PASSWORD_DEFAULT), 
                $data['nama']
            ]);
            
            if (!$userStmt) throw new Exception("Failed to create user");
            
            $userId = $this->db->lastInsertId();
            
            // Step 2: Create guru record
            $guruSql = "INSERT INTO guru (user_id, nip) VALUES (?, ?)";
            $guruStmt = $this->executeQuery($guruSql, [$userId, $data['nip']]);
            
            return $guruStmt ? $this->db->lastInsertId() : false;
            
        } catch (Exception $e) {
            error_log("Add Guru Error: " . $e->getMessage());
            return false;
        }
    }
	// Tambahkan method berikut di bagian GURU MANAGEMENT

	/**
	 * Get guru by ID
	 */
	public function getGuruById($id) {
		$sql = "SELECT g.*, u.nama, u.username, u.id as user_id
				FROM guru g
				JOIN users u ON g.user_id = u.id
				WHERE g.id = ?";
		
		$stmt = $this->executeQuery($sql, [$id]);
		return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
	}

	/**
	 * Update data guru
	 */
	public function updateGuru($id, $data) {
		try {
			// Get current guru data
			$currentData = $this->getGuruById($id);
			if (!$currentData) {
				return false;
			}
			
			// Check if username already exists (excluding current guru)
			$checkSql = "SELECT id FROM users WHERE username = ? AND id != ?";
			$checkStmt = $this->executeQuery($checkSql, [$data['username'], $currentData['user_id']]);
			
			if ($checkStmt && $checkStmt->rowCount() > 0) {
				return false; // Username already exists
			}
			
			// Update user data
			$userSql = "UPDATE users SET nama = ?, username = ? WHERE id = ?";
			$userStmt = $this->executeQuery($userSql, [
				$data['nama'],
				$data['username'],
				$currentData['user_id']
			]);
			
			if (!$userStmt) return false;
			
			// Update password if provided
			if (isset($data['password']) && !empty($data['password'])) {
				$passwordSql = "UPDATE users SET password = ? WHERE id = ?";
				$hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
				$passwordStmt = $this->executeQuery($passwordSql, [
					$hashedPassword,
					$currentData['user_id']
				]);
				
				if (!$passwordStmt) return false;
			}
			
			// Update guru data
			$guruSql = "UPDATE guru SET nip = ? WHERE id = ?";
			$guruStmt = $this->executeQuery($guruSql, [
				$data['nip'],
				$id
			]);
			
			return $guruStmt !== false;
			
		} catch (Exception $e) {
			error_log("Update Guru Error: " . $e->getMessage());
			return false;
		}
	}

	/**
	 * Check if guru has active assignments
	 */
	public function checkGuruAssignment($guruId) {
		$sql = "SELECT COUNT(*) as total FROM guru_mapel WHERE guru_id = ?";
		$stmt = $this->executeQuery($sql, [$guruId]);
		$result = $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : ['total' => 0];
		
		return $result['total'] > 0;
	}

	/**
	 * Delete guru
	 */
	public function deleteGuru($id) {
		try {
			// Start transaction
			$this->db->beginTransaction();
			
			// Get user_id first
			$guruData = $this->getGuruById($id);
			if (!$guruData) {
				throw new Exception("Guru not found");
			}
			
			$userId = $guruData['user_id'];
			
			// Delete from guru_mapel assignments first
			$deleteAssignments = "DELETE FROM guru_mapel WHERE guru_id = ?";
			$stmt1 = $this->db->prepare($deleteAssignments);
			$stmt1->execute([$id]);
			
			// Delete from wali_kelas
			$deleteWaliKelas = "DELETE FROM wali_kelas WHERE guru_id = ?";
			$stmt2 = $this->db->prepare($deleteWaliKelas);
			$stmt2->execute([$id]);
			
			// Then delete the guru record
			$deleteGuru = "DELETE FROM guru WHERE id = ?";
			$stmt3 = $this->db->prepare($deleteGuru);
			$stmt3->execute([$id]);
			
			// Finally delete the user
			$deleteUser = "DELETE FROM users WHERE id = ?";
			$stmt4 = $this->db->prepare($deleteUser);
			$stmt4->execute([$userId]);
			
			// Commit transaction
			$this->db->commit();
			return true;
			
		} catch (Exception $e) {
			// Rollback transaction on error
			$this->db->rollBack();
			error_log("Error deleting guru: " . $e->getMessage());
			return false;
		}
	}

	/**
	 * Check if assignment already exists
	 */
	public function checkExistingAssignment($data) {
		$sql = "SELECT id FROM guru_mapel 
				WHERE guru_id = ? AND mapel_id = ? AND kelas_id = ? AND tahun_ajaran_id = ?";
		$stmt = $this->executeQuery($sql, [
			$data['guru_id'], 
			$data['mapel_id'], 
			$data['kelas_id'], 
			$data['tahun_ajaran_id']
		]);
		
		return $stmt && $stmt->rowCount() > 0;
	}

	/**
	 * Assign guru to mapel
	 */
	public function assignGuruToMapel($data) {
		$sql = "INSERT INTO guru_mapel (guru_id, mapel_id, kelas_id, tahun_ajaran_id) 
				VALUES (?, ?, ?, ?)";
		
		return $this->executeQuery($sql, [
			$data['guru_id'],
			$data['mapel_id'],
			$data['kelas_id'],
			$data['tahun_ajaran_id']
		]);
	}

	/**
	 * Get guru assignments
	 */
	public function getGuruAssignments($guruId) {
		$sql = "SELECT gm.*, m.nama_mapel, m.kode_mapel, k.nama_kelas, k.tingkat, 
					   ta.tahun_ajaran, ta.semester 
				FROM guru_mapel gm
				JOIN mata_pelajaran m ON gm.mapel_id = m.id
				JOIN kelas k ON gm.kelas_id = k.id
				JOIN tahun_ajaran ta ON gm.tahun_ajaran_id = ta.id
				WHERE gm.guru_id = ?
				ORDER BY ta.tahun_ajaran DESC, ta.semester DESC";
		
		$stmt = $this->executeQuery($sql, [$guruId]);
		
		$assignments = [];
		if ($stmt) {
			while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
				$assignments[] = $row;
			}
		}
		return $assignments;
	}
    
    // ==================== KELAS MANAGEMENT ====================
    public function getAllKelas() {
        $sql = "SELECT k.*, 
                       (SELECT COUNT(*) FROM siswa s WHERE s.kelas_id = k.id AND s.status = 'aktif') as total_siswa
                FROM kelas k
                ORDER BY k.tingkat, k.nama_kelas";
        
        $stmt = $this->executeQuery($sql);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    public function addKelas($data) {
        try {
            // Cek duplikasi nama kelas
            $checkSql = "SELECT id FROM kelas WHERE nama_kelas = ?";
            $checkStmt = $this->executeQuery($checkSql, [$data['nama_kelas']]);
            
            if ($checkStmt && $checkStmt->rowCount() > 0) {
                error_log("Duplicate kelas found: " . $data['nama_kelas']);
                return false;
            }
            
            // Insert kelas baru
            $sql = "INSERT INTO kelas (nama_kelas, tingkat, jurusan, kapasitas) VALUES (?, ?, ?, ?)";
            $stmt = $this->executeQuery($sql, [
                $data['nama_kelas'],
                $data['tingkat'],
                $data['jurusan'],
                $data['kapasitas']
            ]);
            
            return $stmt !== false;
            
        } catch (Exception $e) {
            error_log("Error in addKelas: " . $e->getMessage());
            return false;
        }
    }
    
    public function getKelasByNama($namaKelas) {
        $sql = "SELECT * FROM kelas WHERE nama_kelas = ?";
        $stmt = $this->executeQuery($sql, [$namaKelas]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }

    public function getKelasById($kelasId) {
        $sql = "SELECT k.*, 
                       (SELECT COUNT(*) FROM siswa s WHERE s.kelas_id = k.id AND s.status = 'aktif') as total_siswa
                FROM kelas k
                WHERE k.id = ?";
        
        $stmt = $this->executeQuery($sql, [$kelasId]);
        return $stmt ? $stmt->fetch(PDO::FETCH_ASSOC) : false;
    }

    public function updateKelas($kelasId, $data) {
        $sql = "UPDATE kelas SET nama_kelas = ?, tingkat = ?, jurusan = ?, kapasitas = ? WHERE id = ?";
        
        return $this->executeQuery($sql, [
            $data['nama_kelas'],
            $data['tingkat'],
            $data['jurusan'],
            $data['kapasitas'],
            $kelasId
        ]);
    }

    public function deleteKelas($kelasId) {
        try {
            // Check if kelas has students
            $checkSiswa = $this->executeQuery(
                "SELECT COUNT(*) as total FROM siswa WHERE kelas_id = ? AND status = 'aktif'",
                [$kelasId]
            );
            
            if ($checkSiswa && $checkSiswa->fetch(PDO::FETCH_ASSOC)['total'] > 0) {
                return false; // Cannot delete kelas with active students
            }
            
            // Delete related records first
            $this->executeQuery("DELETE FROM wali_kelas WHERE kelas_id = ?", [$kelasId]);
            $this->executeQuery("DELETE FROM guru_mapel WHERE kelas_id = ?", [$kelasId]);
            
            // Then delete kelas
            $result = $this->executeQuery("DELETE FROM kelas WHERE id = ?", [$kelasId]);
            
            return $result !== false;
            
        } catch (Exception $e) {
            error_log("Delete Kelas Error: " . $e->getMessage());
            return false;
        }
    }
    
    // ==================== MATA PELAJARAN ====================
// ==================== MATA PELAJARAN ====================
	public function getAllMapel() {
		$sql = "SELECT * FROM mata_pelajaran ORDER BY kode_mapel";
		$stmt = $this->executeQuery($sql);
		return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
	}

	public function addMapel($kode_mapel, $nama_mapel, $deskripsi = '') {
		try {
			$sql = "INSERT INTO mata_pelajaran (kode_mapel, nama_mapel, deskripsi) VALUES (?, ?, ?)";
			$stmt = $this->db->prepare($sql);
			return $stmt->execute([$kode_mapel, $nama_mapel, $deskripsi]);
		} catch (PDOException $e) {
			error_log("Database Error in addMapel: " . $e->getMessage());
			return false;
		}
	}

	public function updateMapel($id, $kode_mapel, $nama_mapel, $deskripsi = '') {
		try {
			$sql = "UPDATE mata_pelajaran SET kode_mapel = ?, nama_mapel = ?, deskripsi = ? WHERE id = ?";
			$stmt = $this->db->prepare($sql);
			return $stmt->execute([$kode_mapel, $nama_mapel, $deskripsi, $id]);
		} catch (PDOException $e) {
			error_log("Database Error in updateMapel: " . $e->getMessage());
			return false;
		}
	}

	public function deleteMapel($id) {
		try {
			$sql = "DELETE FROM mata_pelajaran WHERE id = ?";
			$stmt = $this->db->prepare($sql);
			return $stmt->execute([$id]);
		} catch (PDOException $e) {
			error_log("Database Error in deleteMapel: " . $e->getMessage());
			return false;
		}
	}

	public function getMapelById($id) {
		try {
			$sql = "SELECT * FROM mata_pelajaran WHERE id = ?";
			$stmt = $this->db->prepare($sql);
			$stmt->execute([$id]);
			return $stmt->fetch(PDO::FETCH_ASSOC);
		} catch (PDOException $e) {
			error_log("Database Error in getMapelById: " . $e->getMessage());
			return false;
		}
	}
    
    // ==================== GURU MAPEL (PENUGASAN) ====================
    public function getGuruMapelByTahun($tahunAjaranId) {
        $sql = "SELECT gm.*, g.nip, u.nama as nama_guru, m.nama_mapel, m.kode_mapel, 
                       k.nama_kelas, k.tingkat, k.jurusan, ta.tahun_ajaran, ta.semester
                FROM guru_mapel gm
                JOIN guru g ON gm.guru_id = g.id
                JOIN users u ON g.user_id = u.id
                JOIN mata_pelajaran m ON gm.mapel_id = m.id
                JOIN kelas k ON gm.kelas_id = k.id
                JOIN tahun_ajaran ta ON gm.tahun_ajaran_id = ta.id
                WHERE gm.tahun_ajaran_id = ?
                ORDER BY k.tingkat, k.nama_kelas, m.nama_mapel";
        
        $stmt = $this->executeQuery($sql, [$tahunAjaranId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
    
    public function assignGuruMapel($data) {
        $sql = "INSERT INTO guru_mapel (guru_id, mapel_id, kelas_id, tahun_ajaran_id) 
                VALUES (?, ?, ?, ?)";
        
        return $this->executeQuery($sql, [
            $data['guru_id'],
            $data['mapel_id'],
            $data['kelas_id'],
            $data['tahun_ajaran_id']
        ]);
    }
    
    public function removeGuruMapel($assignmentId) {
        $sql = "DELETE FROM guru_mapel WHERE id = ?";
        return $this->executeQuery($sql, [$assignmentId]);
    }
    
    // ==================== WALI KELAS MANAGEMENT ====================
    
	     public function getWaliKelasByTahun($tahun_ajaran_id)
    {
        $sql = "SELECT wk.*, 
                       g.nama as nama_guru, g.nip, g.status as status_guru,
                       k.nama_kelas, k.tingkat, k.jurusan, k.status as status_kelas,
                       ta.tahun_ajaran, ta.semester, ta.status as status_tahun
                FROM wali_kelas wk
                JOIN guru g ON wk.guru_id = g.id
                JOIN kelas k ON wk.kelas_id = k.id
                JOIN tahun_ajaran ta ON wk.tahun_ajaran_id = ta.id
                WHERE wk.tahun_ajaran_id = ?
                ORDER BY k.tingkat, k.nama_kelas";
        
        return $this->db->query($sql, [$tahun_ajaran_id])->fetchAll();
    }

    /**
     * Get wali kelas by kelas dan tahun ajaran
     */
    public function getWaliKelasByKelas($kelas_id, $tahun_ajaran_id)
    {
        $sql = "SELECT wk.*, g.nama as nama_guru, g.nip, k.nama_kelas, ta.semester
                FROM wali_kelas wk
                JOIN guru g ON wk.guru_id = g.id
                JOIN kelas k ON wk.kelas_id = k.id
                JOIN tahun_ajaran ta ON wk.tahun_ajaran_id = ta.id
                WHERE wk.kelas_id = ? AND wk.tahun_ajaran_id = ?";
        
        return $this->db->query($sql, [$kelas_id, $tahun_ajaran_id])->fetch();
    }

    /**
     * Cek apakah guru sudah menjadi wali di kelas lain di semester yang sama
     */
    public function isGuruAlreadyWali($guru_id, $tahun_ajaran_id, $semester, $exclude_wali_id = null)
    {
        $sql = "SELECT COUNT(*) as total 
                FROM wali_kelas 
                WHERE guru_id = ? 
                AND tahun_ajaran_id = ? 
                AND semester = ?";
        
        $params = [$guru_id, $tahun_ajaran_id, $semester];
        
        if ($exclude_wali_id) {
            $sql .= " AND id != ?";
            $params[] = $exclude_wali_id;
        }
        
        $result = $this->db->query($sql, $params)->fetch();
        return $result['total'] > 0;
    }

    /**
     * Assign wali kelas baru
     */
    public function assignWaliKelas($data)
    {
        try {
            $this->db->beginTransaction();

            // Cek apakah kelas sudah memiliki wali di semester ini
            $existingWali = $this->getWaliKelasByKelas($data['kelas_id'], $data['tahun_ajaran_id']);
            if ($existingWali) {
                throw new Exception("Kelas ini sudah memiliki wali kelas di semester " . $existingWali['semester']);
            }

            // Cek apakah guru sudah menjadi wali di kelas lain di semester yang sama
            if ($this->isGuruAlreadyWali($data['guru_id'], $data['tahun_ajaran_id'], $data['semester'])) {
                throw new Exception("Guru ini sudah menjadi wali kelas di kelas lain di semester yang sama");
            }

            $sql = "INSERT INTO wali_kelas (guru_id, kelas_id, tahun_ajaran_id, semester, created_by) 
                    VALUES (?, ?, ?, ?, ?)";
            
            $params = [
                $data['guru_id'],
                $data['kelas_id'],
                $data['tahun_ajaran_id'],
                $data['semester'],
                Session::get('user_id')
            ];
            
            $result = $this->db->query($sql, $params);
            
            $this->db->commit();
            return $result;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Error assignWaliKelas: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Update wali kelas
     */
    public function updateWaliKelas($id, $data)
    {
        try {
            $this->db->beginTransaction();

            // Cek apakah guru sudah menjadi wali di kelas lain di semester yang sama
            if ($this->isGuruAlreadyWali($data['guru_id'], $data['tahun_ajaran_id'], $data['semester'], $id)) {
                throw new Exception("Guru ini sudah menjadi wali kelas di kelas lain di semester yang sama");
            }

            $sql = "UPDATE wali_kelas 
                    SET guru_id = ?, updated_at = NOW() 
                    WHERE id = ?";
            
            $result = $this->db->query($sql, [$data['guru_id'], $id]);
            
            $this->db->commit();
            return $result;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Error updateWaliKelas: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Remove wali kelas
     */
    public function removeWaliKelas($id)
    {
        try {
            $sql = "DELETE FROM wali_kelas WHERE id = ?";
            return $this->db->query($sql, [$id]);
            
        } catch (Exception $e) {
            error_log("Error removeWaliKelas: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get all guru yang aktif

    public function getAllGuru()
    {
        $sql = "SELECT id, nama, nip, status FROM guru WHERE status = 'aktif' ORDER BY nama";
        return $this->db->query($sql)->fetchAll();
    }

    /**
     * Get all kelas yang aktif
     
    public function getAllKelas()
    {
        $sql = "SELECT id, nama_kelas, tingkat, jurusan, status FROM kelas WHERE status = 'aktif' ORDER BY tingkat, nama_kelas";
        return $this->db->query($sql)->fetchAll();
    }

    /**
     * Get active tahun ajaran
     
    public function getActiveTahunAjaran()
    {
        $sql = "SELECT * FROM tahun_ajaran WHERE status = 'aktif' LIMIT 1";
        return $this->db->query($sql)->fetch();
    }

    /**
     * Get wali kelas by ID
     */
    public function getWaliKelasById($id)
    {
        $sql = "SELECT wk.*, g.nama as nama_guru, k.nama_kelas, ta.tahun_ajaran, ta.semester
                FROM wali_kelas wk
                JOIN guru g ON wk.guru_id = g.id
                JOIN kelas k ON wk.kelas_id = k.id
                JOIN tahun_ajaran ta ON wk.tahun_ajaran_id = ta.id
                WHERE wk.id = ?";
        
        return $this->db->query($sql, [$id])->fetch();
    }
	
    
    // ==================== KELAS SISWA MANAGEMENT ====================
    public function getSiswaByKelas($kelasId) {
        $sql = "SELECT s.*, u.nama, u.username
                FROM siswa s
                JOIN users u ON s.user_id = u.id
                WHERE s.kelas_id = ? AND s.status = 'aktif'
                ORDER BY u.nama";
        
        $stmt = $this->executeQuery($sql, [$kelasId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }

    public function moveSiswaToKelas($siswaId, $newKelasId, $tahunAjaranId) {
        try {
            // Update siswa's current class
            $sql = "UPDATE siswa SET kelas_id = ? WHERE id = ?";
            $result = $this->executeQuery($sql, [$newKelasId, $siswaId]);
            
            if ($result) {
                // Add to riwayat_kelas
                $tahunAktif = $this->getActiveTahunAjaran();
                if ($tahunAktif) {
                    $riwayatSql = "INSERT INTO riwayat_kelas (siswa_id, kelas_id, tahun_ajaran_id, semester) 
                                   VALUES (?, ?, ?, ?)";
                    $this->executeQuery($riwayatSql, [
                        $siswaId, 
                        $newKelasId, 
                        $tahunAjaranId,
                        $tahunAktif['semester']
                    ]);
                }
                return true;
            }
            return false;
        } catch (Exception $e) {
            error_log("Move Siswa Error: " . $e->getMessage());
            return false;
        }
    }

    public function removeSiswaFromKelas($siswaId) {
        $sql = "UPDATE siswa SET kelas_id = NULL WHERE id = ?";
        return $this->executeQuery($sql, [$siswaId]);
    }

    // ==================== KELAS MAPEL MANAGEMENT ====================
    public function getMapelByKelas($kelasId, $tahunAjaranId) {
        $sql = "SELECT gm.*, g.nip, u.nama as nama_guru, m.nama_mapel, m.kode_mapel, 
                       k.nama_kelas, ta.tahun_ajaran, ta.semester as tahun_semester
                FROM guru_mapel gm
                JOIN guru g ON gm.guru_id = g.id
                JOIN users u ON g.user_id = u.id
                JOIN mata_pelajaran m ON gm.mapel_id = m.id
                JOIN kelas k ON gm.kelas_id = k.id
                JOIN tahun_ajaran ta ON gm.tahun_ajaran_id = ta.id
                WHERE gm.kelas_id = ? AND gm.tahun_ajaran_id = ?
                ORDER BY m.nama_mapel";
        
        $stmt = $this->executeQuery($sql, [$kelasId, $tahunAjaranId]);
        return $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
    }
}
?>